/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.java;

import java.util.LinkedHashMap;


/**
 * This bridge loader isolates the Java component class loaders
 * from the slow JRE loaders by caching unsuccessful class loading
 * attempts (that are handled very inefficiently by the JRE loaders)
 * 
 * @author hb
 *
 */
public class SpeedValveLoader extends ClassLoader {

	public SpeedValveLoader(ClassLoader parent) {
		super(parent);
	}

	/**
	 * An LRU "not to be found" classes cache (speed up of negative lookups)
	 */
	@SuppressWarnings("serial")
	private static class NotFoundClassesCache extends LinkedHashMap<String,Boolean> {
		@Override
		protected boolean removeEldestEntry(java.util.Map.Entry<String, Boolean> eldest) {
			return this.size()>600;
		}
	
		public boolean has(String name) {
			return this.containsKey(name);
		}
			
		public void put(String name) {
			this.remove(name);
			this.put(name,Boolean.TRUE);
		}
	}

	private NotFoundClassesCache notFound = new NotFoundClassesCache();

	
	@Override
	protected synchronized Class<?> loadClass(String name, boolean resolve) throws ClassNotFoundException {
		// check in "negative" cache
		if (notFound.has(name)) {
			notFound.put(name);
			throw new ClassNotFoundException(name);
		}
		try {
			return super.loadClass(name, resolve);
		} catch (ClassNotFoundException cnfe) {
			notFound.put(name);
			throw cnfe;
		}
	}
}
