/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 *
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.resources;

import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Queue;
import java.util.WeakHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.management.ObjectName;
import javax.management.StandardMBean;

import com.zfabrik.resources.IResourceHandle;
import com.zfabrik.resources.IResourceInfo;
import com.zfabrik.resources.IResourceLookup;
import com.zfabrik.resources.ResourceBusyException;
import com.zfabrik.resources.TypeRef;
import com.zfabrik.resources.provider.IResourceProvider;
import com.zfabrik.resources.provider.Resource;
import com.zfabrik.work.CycleProtector;

/**
 * Implementation of instance management per namespace. We make use of a
 * resource id level locking mechanim that is supported by the resource handle
 * implementation.
 *
 * Note that we may optimistically create a yet to be assigned resource handle
 * that represents a mere lock - until it gets its backing resource assigned.
 *
 * The general challenge is to clean out locks and resources. The general
 * solution is:
 * <ul>
 * <li>We do not lock at all when calling life-cycle methods</li>
 * <li>The VH is used as mutex for dependency handling (owns its dependencies)
 * <li>The RH is used as mutex for the actual resource assignment</li>
 * <li>Dependencies are managed by ValueHolder. The ValueHolder is a unique key
 * for its potentially claimed resource instance.</lI>
 * </ul>
 *
 * @author hb
 *
 */
public class NamespaceImpl implements IResourceLookup {

	/**
	 * This enum is used internally to classify the style of resource lookup. Either a query or an actual get, either using the 
	 * type ref API or the class based API. 
	 */
	public enum LookupType {
		QUERY_BY_CLASS(true,true), 
		QUERY_BY_TYPE_REF(false,true), 
		GET_BY_CLASS(true,false), 
		GET_BY_TYPE_REF(false,false);
		
		private boolean byClass;
		private boolean query;
		
		private LookupType(boolean byClass, boolean query) {
			this.byClass = byClass;
			this.query = query;
		}
		
		public boolean isByClass() {
			return byClass;
		}
		public boolean isQuery() {
			return query;
		}
	}

	
	// only used in testing
	public boolean ticking = true;

	// keep resource handles uniquely by name
	// but allow collection (a smart weak value map)
	private ResourcesMap resources;
	// keep resource handles by weak keys. In general
	// Objects passed out from resources that should
	// keep the resource and handle alive
	private WeakHashMap<Object, ResourceHandleImpl> attachments;
	private IResourceProvider prov;
	private String namespace;
	private Queue<ResourcesMap.ValueHolder> expQ, invQ;

	// JMX
	private ObjectName on;
	private NamespaceMBeanImpl mb;

	//
	private class NamespaceMBeanImpl implements NamespaceMBean {
		private Date resetTime = new Date();
		private long numCleanUps;
		private long numProviderQueries;
		private long numQueries;
		private long numSuccessfulQueries;

		public int getNumShrinks() {
			synchronized (NamespaceImpl.this.resources) {
				return NamespaceImpl.this.resources.getShrinks();
			}
		}

		public int getCurrentSize() {
			synchronized (NamespaceImpl.this.resources) {
				return NamespaceImpl.this.resources.map().size();
			}
		}

		public Date getLastResetTime() {
			synchronized (NamespaceImpl.this.resources) {
				return this.resetTime;
			}
		}

		public int getMaxSize() {
			synchronized (NamespaceImpl.this.resources) {
				return NamespaceImpl.this.resources.getMaxSize();
			}
		}

		public long getNumCleanUps() {
			synchronized (NamespaceImpl.this.resources) {
				return this.numCleanUps;
			}
		}

		public long getNumProviderQueries() {
			synchronized (NamespaceImpl.this.resources) {
				return this.numProviderQueries;
			}
		}

		public long getNumQueries() {
			synchronized (NamespaceImpl.this.resources) {
				return this.numQueries;
			}
		}

		public long getNumSuccessfulQueries() {
			synchronized (NamespaceImpl.this.resources) {
				return this.numSuccessfulQueries;
			}
		}

		public List<String> getAllResourceNames() {
			synchronized (NamespaceImpl.this.resources) {
				return NamespaceImpl.this.list();
			}
		}

		public void reset() {
			synchronized (NamespaceImpl.this.resources) {
				this.numCleanUps = 0;
				this.numProviderQueries = 0;
				this.numQueries = 0;
				this.numSuccessfulQueries = 0;
				NamespaceImpl.this.resources.resetStats();
				this.resetTime = new Date();
			}
		}
	}

	//

	public NamespaceImpl(String namespace, IResourceProvider prov) {
		this.prov = prov;
		this.namespace = namespace;
		this.invQ = new LinkedBlockingQueue<ResourcesMap.ValueHolder>();
		this.expQ = new LinkedBlockingQueue<ResourcesMap.ValueHolder>();
		this.resources = new ResourcesMap();
		this.resources.setExpirationQueue(this.expQ);
		this.resources.setInvalidationQueue(this.invQ);
		try {
			this.on = ObjectName.getInstance("zfabrik:type=" + NamespaceImpl.class.getName() + ",name=" + this.namespace);
			this.mb = new NamespaceMBeanImpl();
			ManagementFactory.getPlatformMBeanServer().registerMBean(new StandardMBean(this.mb, NamespaceMBean.class, false), this.on);
		} catch (Exception e) {
			throw new IllegalStateException("Failed to register MBean", e);
		}
	}
	
	/*
	 * Resolve by resourceId and typeref. Conditionally only query and/or call non-parameterized API
	 */
	protected <T> T resolve(String resourceId, TypeRef<T> typeRef, LookupType lookupType) {
		return CycleProtector.supply("lookup:"+resourceId+"/"+typeRef.getType(),()->{ 
			ResourceHandleImpl rh;
			if (lookupType.isQuery()) {
				rh = _queryResourceHandle(resourceId);
			} else {
				rh = _getResourceHandleImplNonNull(resourceId);
				synchronized (rh) {
					// check if the actual resource was assigned
					if (!rh._isAssigned()) {
						// need to get one
						Resource r = this.prov.get(resourceId);
						if (r != null)
							rh._assign(r);
						synchronized (this.resources) {
							this.mb.numProviderQueries++;
						}
					}
				}
			}
			if (rh!=null) {
				ResourceHandleImpl r2 = rh.toOutside();
				if (r2 != null) {
					T r = r2.as(typeRef,lookupType);
					synchronized (this.resources) {
						if (r != null) {
							this.mb.numSuccessfulQueries++;
						}
						this.mb.numQueries++;
					}
					return r;
				}
			}
			return null;
	    });
	}

	@Override
	public <T> T lookup(String resourceId, Class<T> clz) {
		return resolve(resourceId,new SimpleTypeRef<T>(clz), LookupType.GET_BY_CLASS);
	}

	@Override
	public <T> T lookup(String resourceId, TypeRef<T> typeRef) {
		return resolve(resourceId, typeRef, LookupType.GET_BY_TYPE_REF);
	}

	@Override
	public <T> T query(String resourceId, Class<T> clz) {
		return resolve(resourceId,new SimpleTypeRef<T>(clz), LookupType.QUERY_BY_CLASS);
	}

	@Override
	public <T> T query(String resourceId, TypeRef<T> typeRef) {
		return resolve(resourceId, typeRef, LookupType.QUERY_BY_TYPE_REF);
	}
	
	public List<String> list() {
		synchronized (this.resources) {
			return new ArrayList<String>(this.resources.map().keySet());
		}
	}

	public IResourceHandle put(String name, Resource resource) {
		tick();
		if (resource != null) {
			ResourceHandleImpl rh = _getResourceHandleImplNonNull(name);
			synchronized (rh) {
				rh._assign(resource);
				return rh;
			}
		}
		throw new IllegalArgumentException("Cannot bind null resource by name " + name + " in namespace " + namespace);
	}

	// piggybacked, async cleanup
	// and general GC handling
	// Makes sure that collected handles will have a decent dependency
	// invalidation
	// before new handles get passed out.
	// Together with get/query res handle, we can get into situations where
	// invalidate must be executed concurrently
	// by several threads as to make sure it does not get blocked and threads do
	// not end up in busy wait loops.
	// this is somewhat tricky. We memorize the currently being invalidated vh
	// and
	// let it be picked up concurrently if the queues
	// don't yield anything to work on.
	private ResourcesMap.ValueHolder _inInvalidation;

	private void tick() {
		if (!ticking) {
			return; // only for testing
		}
		int c = 0;
		synchronized (this.resources) {
			this.resources.tick();
			if (this.attachments!=null) {
				// expunging stale entries is a side effect
				// of asking for the size of a weak map
				this.attachments.size();
			}
		}
		// concurrently run expirations and invalidations
		ResourcesMap.ValueHolder vh;
		do {
			synchronized (this.resources) {
				if (this._inInvalidation != null) {
					vh = this._inInvalidation;
				} else {

					do {
						vh = this.invQ.poll();
						// unless there is nothing or what we found is still in the map (because otherwise
						// it would have been invalidated) we keep polling
					} while (vh!=null && !this.resources.map().containsKey(vh.getKey()));

					if (vh == null) {
						vh = this.expQ.poll();
					}
					if (vh != null) {
						this._inInvalidation = vh;
						c++;
					}
				}
			}
			if (vh != null) {
				if (logger.isLoggable(Level.FINE)) {
					logger.fine("Tick clean out of " + this.namespace + "/" + vh.getKey());
				}
				_invalidate(vh.getExtra(), true);
				synchronized (this.resources) {
					this._inInvalidation = null; // remove
					// eventually we can actually remove the resource
					this.resources.remove(vh.getKey());
				}
			}
		} while (vh != null);
		if (c != 0) {
			synchronized (this.resources) {
				this.mb.numCleanUps++;
			}
		}
	}

	// retrieve a possibly incomplete resource handle
	private ResourceHandleImpl _getResourceHandleImplNonNull(String resourceId) {
		ResourceHandleImpl rh = null;
		// memorize whether we are already tick looping to get rid
		// of gc'd dependencies
		boolean looping=false;
		do {
			tick();
			synchronized (this.resources) {
				ResourcesMap.ValueHolder vh = this.resources.map().get(resourceId);
				if (vh != null) {
					rh = vh.getValue();
					// if null here, the handle was GC'd.
					// if we make sure that the cleanout happens before we pass out anything new.
					// it doesn't happen right away, we force it by pushing the holder up the invalidation queue and running yet
					// another tick() round. This is a safety measure as the GC might nor put stuff in invalidation
					// queues right away (not required by spec) - but we will give it one try without putting it
					// into the queue (as normally that suffices and the contains check might be expensive).
					if (looping && !this.invQ.contains(vh)) {
						this.invQ.add(vh);
					}
					// note that we are taking an extra loop.
					looping = true;
				} else {
					rh = new ResourceHandleImpl();
					vh = this.resources.put(resourceId, rh);
					rh.env = new ResourceEnv(this, vh);
					vh.setExtra(rh.env);
				}
			}
		} while (rh == null);
		return rh;
	}

	private ResourceHandleImpl _queryResourceHandle(String name) {
		ResourceHandleImpl rh = null;
		do {
			tick();
			synchronized (this.resources) {
				ResourcesMap.ValueHolder vh = this.resources.map().get(name);
				if (vh != null) {
					rh = vh.getValue();
				} else {
					// ok, there simply is none
					return null;
				}
			}
		} while (rh == null);
		return rh;
	}

	/**
	 * Invalidate a resource because of a dependency in svh.
	 *
	 * Following the general sync strategy we cannot simply lock from dependency to dependant. Hence we
	 * make sure we hit all dependants at least once, kick the resource and just make sure once more there is no
	 * dependant left.
	 *
	 * <em>previously:</em>
	 * Also, sync'd on the handle (see also {@link ResourceHandleImpl#as(Class)}) we test whether a dependency
	 * is still in effect (weak invalidations may be scheduled late - while an as() call has already been performed
	 * that changed dependencies effectively. There is still a small gap though and a resource should always be ready
	 * to get up again. At least for direct weak dependencies we can do something...
	 *
	 * <b>NOTE:</b> We cannot sync on the handle, as the resources may have their own synchronization and resources
	 * call RM which calls resources. So in short this is super deadlock prone!. We had a 12h outage on Aug. 04, 2009 due to
	 * a deadlock on this sync!!
	 *
	 *
	 */
	protected void _invalidate(ResourceEnv env, boolean forced) {
		ResourceHandleImpl rh = env.vh.getValue();
		if (rh != null) {
			if (logger.isLoggable(Level.FINE)) {
				logger.fine("Invalidating resource " + this.namespace + "/" + env.vh.getKey());
			}
			_invalidateDependants(env, forced);
			boolean busyKeep = false;
			try {
				rh._invalidateLocal(forced);
			} catch (ResourceBusyException rbe) {
				if (!forced) {
					busyKeep = true;
					throw new ResourceBusyException("Resource " + env.vh.getKey() + " is indirectly busy and cannot be invalidated at this time", rbe);
				} else {
					logger.log(Level.WARNING, "Ignoring busy resource condition because of forced invalidation", rbe);
				}
			} catch (Exception e) {
				logger.log(Level.WARNING, "Caught exception upon resource invalidation (" + env.vh.getKey() + ")", e);
			} finally {
				if (!busyKeep) {
					env.removeDeps();
				}
			}
		} else {
			// just a cleanup! The resource is gone. However,
			// it may be in some dependants set, so we should
			// remove it from there. We DO NOT INVALIDATE dependants
			// as we keep strong refs to dependencies. In other words
			// if there are still dependants that is an error. But as they
			// may have been collected as well the process of removing
			// the dependency may happen asynchronously.
			// Note there is no need to worry about resurrections,
			// as all read and write accesses that would lead to such
			// will call tick() before and hence a new resource and
			// value holder will be instantiated
			env.removeDeps();
		}
	}

	// invalidate all dependants at least once
	private void _invalidateDependants(ResourceEnv env, boolean forced) {
		ResourceEnv dep;
		// until there is no dependant left, we will kick them all
		// so that at least once everything necessary has been
		// dropped
		while ((dep = env.popDependant())!=null) {
			dep.ns._invalidate(dep, forced);
		};
	}

	public void release(boolean forced) {
		// release all resources.
		//
		try {
			List<ResourcesMap.ValueHolder> vhs;
			synchronized (this.resources) {
				Collection<ResourcesMap.ValueHolder> x = this.resources.map().values();
				vhs = new ArrayList<ResourcesMap.ValueHolder>(x.size());
				vhs.addAll(x);
			}
			for (ResourcesMap.ValueHolder vh : vhs) {
				_invalidate(vh.getExtra(), forced);
			}
			tick();
		} finally {
			try {
				ManagementFactory.getPlatformMBeanServer().unregisterMBean(this.on);
			} catch (Exception e) {
				logger.log(Level.WARNING, "Error during MBean unregistration", e);
			}
		}
	}

	protected IResourceInfo getResourceInfo(ResourcesMap.ValueHolder vh) {
		return new ResourceInfoImpl(vh.getKey(), vh.getTtl(), vh.getExpiration(), vh.getRefMode());
	}


	private final static Logger logger = ResourceManagerImpl.logger;

	public void adjust(ResourcesMap.ValueHolder vh, long ttl, long exp, short rm) {
		synchronized (this.resources) {
			vh.adjust(ttl, exp, rm);
		}
	}

	// attach, detach resource to object

	protected synchronized void attach(Object o, ResourceHandleImpl rhi) {
		if (this.attachments==null) {
			this.attachments = new WeakHashMap<Object, ResourceHandleImpl>();
		}
		this.attachments.put(o,rhi);
	}

	protected synchronized void detach(Object o) {
		if (this.attachments!=null) {
			this.attachments.remove(o);
			if (this.attachments.size()==0) {
				this.attachments = null;
			}
		}
	}

}
