/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.work;

import java.lang.management.ManagementFactory;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import javax.management.ObjectName;

import com.zfabrik.impl.work.jmx.ThreadPool;
import com.zfabrik.work.IThreadPool;
import com.zfabrik.work.WorkUnit;

public class ThreadPoolImpl implements IThreadPool {
	private WorkManagerImpl wm;
	private Logger logger, perfLogger;
	private String id;
	public int conc; // current concurrency
	
	/**
	 * This is the specified maximal concurrency. It is observed
	 * when processing asynchronous work. When blocking work from a thread of this pool, 
	 * we grant some excess to avoid potential thread starvation.  
	 */
	public int maxconc; // hard limit: max concurrency
	
	/**
	 * Extra granted excess count of threads 
	 * to make sure a blocking execution may not lead to starvation.
	 */
	public int excess;

	public List<Runnable> tasks = new LinkedList<Runnable>();
	private ObjectName on;

	
	// actually waiting foreign threads
	public Set<Thread> foreignPoolThreads = Collections.synchronizedSet(new HashSet<Thread>());

	// stats
	public Date lastReset = new Date();
	public int maxAchievedConcurrency = 0;
	public int maxTaskQueueLength = 0;
	public int currentQueueLength = 0;
	public int currentConcurrency = 0;
	public long tasksCompleted = 0;
	public long lastDispatch = 0;

	public ThreadPoolImpl(String id, WorkManagerImpl wm) {
		this.id = id;
		this.wm = wm;
		this.conc = 0;
		this.maxconc = 1;
		this.logger = Logger.getLogger("system.work.threadpool." + id);
		this.perfLogger = Logger.getLogger("system.performance.system.work.threadpool." + id);
		try {
			this.on = ObjectName.getInstance("zfabrik:type="+ThreadPoolImpl.class.getName()+",name=" + this.id);
			ManagementFactory.getPlatformMBeanServer().registerMBean(new ThreadPool(this), this.on);
		} catch (Exception e) {
			throw new IllegalStateException("mbean registration for thread pool failed", e);
		}
	}

	protected void shutdown() {
		try {
			ManagementFactory.getPlatformMBeanServer().unregisterMBean(this.on);
		} catch (Exception e) {
			logger.log(Level.WARNING, "Mbean unregistration error", e);
		}
	}

	public void setMaxConcurrency(int maxconcurrency) {
		logger.fine("setting max concurrency of thread pool " + this.id + " from " + this.maxconc + " to " + maxconcurrency);
		this.maxconc = maxconcurrency;
	}
	
	public int getMaxConcurrency() {
		return this.maxconc;
	}
	
	public int getMaxAchievedConcurrency() {
		return maxAchievedConcurrency;
	}

	public void execute(boolean block, Runnable... runnables) {
		this.execute(block, Arrays.asList(runnables));
	}

	public boolean isPoolThread(Thread t) {
		return  ((t instanceof WorkerThread) && (((WorkerThread) t).getThreadpool() == this) || this.foreignPoolThreads.contains(t));
	}

/*
Our own little latch implementation. Used java.util.concurrency.CountDownLatch before and had a problem that turned out to be my own problem anyway. So, we could easily revert to 
the java.util.concurrency solution.
 */
	private static class Latch {
		private int c;
		public Latch(int c) {
			this.c=c;
		}
		public synchronized void countDown() {
			if (c>0) this.c--;
			if (c==0) this.notify();
		}
		public synchronized void await() throws InterruptedException {
			while (this.c>0) {
				this.wait();
			}
		}
	}
	
	public void execute(boolean block, final Collection<? extends Runnable> runnables) {
		if (runnables.size()<1) return;
		if (logger.isLoggable(Level.FINER)) {
			logger.finer("putting collection of " + runnables.size() + " into execution queue");
		}
		// check whether this thread is to participate and we must block.
		boolean pt = isPoolThread(Thread.currentThread());
		if (block) {
			Latch cdl = new Latch(runnables.size());
			try {
				synchronized (this) {
					if (pt) {
						// #2040: grant some excess so we may not starve
						this.excess++;
					}
					_dispatch(wrap(runnables, cdl::countDown));
				}
				try {
					// wait to enforce blocking strategy
					cdl.await();
				} catch (InterruptedException e) {
					logger.log(Level.WARNING, "unexpected thread interruption!", e);
				}
			} finally {
				if (pt) {
					synchronized (this) {
						// remove granted excess
						this.excess--;
					}
				}
			}
		} else {
			// this is the simpler case. Simply drop all into the task queue
			synchronized (this) {
				_dispatch(wrap(runnables,()->{}));
			}
		}
	}

	/**
	 * Wrap runnables to use the calling thread's classloader
	 * @param runnables
	 * @param c
	 * @param cdl
	 * @return
	 */
	private LinkedList<Runnable> wrap(final Collection<? extends Runnable> runnables, Runnable oneDown) {
		ClassLoader ccl = Thread.currentThread().getContextClassLoader();
		return  
			runnables.stream().map(todo-> (Runnable) ()->{
				// use calling thread's ccl
				ClassLoader ocl = Thread.currentThread().getContextClassLoader();
				Thread.currentThread().setContextClassLoader(ccl);
				try {
					doIt(todo);
				} finally {
					Thread.currentThread().setContextClassLoader(ocl);
					if (oneDown!=null) { oneDown.run(); }
				}
			})
			.collect(Collectors.toCollection(LinkedList::new));
	}

	// dispatch a number of tasks
	private void _dispatch(final Collection<? extends Runnable> runnables) {
		this.lastDispatch = System.currentTimeMillis();
		tasks.addAll(runnables);
		this.currentQueueLength = tasks.size();
		if (this.currentQueueLength > this.maxTaskQueueLength)
			this.maxTaskQueueLength = this.currentQueueLength;
		_pickUpWork();
	}

	private void _dispatch(Runnable runnable) {
		this.lastDispatch = System.currentTimeMillis();
		tasks.add(runnable);
		this.currentQueueLength = tasks.size();
		if (this.currentQueueLength > this.maxTaskQueueLength)
			this.maxTaskQueueLength = this.currentQueueLength;
		_pickUpWork();
	}

	// let the pool pick up some tasks
	private void _pickUpWork() {
		// make sure we use all there is
		while ((this.conc < (this.maxconc+this.excess)) && (!this.tasks.isEmpty())) {
			this.conc++;
			this.wm.start(this, this.tasks.remove(0));
		}
		if (this.conc > this.maxAchievedConcurrency)
			this.maxAchievedConcurrency = this.conc;
	}

	/**
	 * pulls a task from the queue
	 * 
	 * @return
	 */
	protected synchronized Runnable fetchTask() {
		if (!this.tasks.isEmpty())
			return this.tasks.remove(0);
		return null;
	}

	/**
	 * pulls a task from the queue and reduces conc if none avail
	 * 
	 * @return
	 */
	protected synchronized Runnable fetchTaskOrQuit() {
		Runnable r = fetchTask();
		this.currentQueueLength = tasks.size();
		if (r == null) {
			_decreaseConc();
		}
		return r;
	}

	private void _decreaseConc() {
		this.conc--; // thread will stop or return to pool
	}
	
	private void _increaseConc() {
		this.conc++;
		if (this.conc > this.maxAchievedConcurrency)
			this.maxAchievedConcurrency = this.conc;
	}

	/**
	 * NOTE THERE ARE DUPLICATIONS BENEATH! KEEP THEM IN SYNC!
	 */
	
	
	/**
	 * execute a callable in the name of this threadpool
	 */
	protected <T> T doIt(Callable<T> c) throws Exception {
		long start = System.currentTimeMillis();
		if (logger.isLoggable(Level.FINER)) {
			logger.finer("executing " + c);
		}
		try {
			return WorkUnit.work(c);
		} finally {
			synchronized (this) {
				this.tasksCompleted++;
			}
			if (perfLogger.isLoggable(Level.FINE)) {
				long durationTotal = System.currentTimeMillis() - start;
				perfLogger.fine("system.work.threadpool." + id + ",total work unit," + c + ",duration," + durationTotal + ",ms");
				perfLogger.fine("system.work.threadpool." + id + ",total runnable," + c + ",duration," + durationTotal + ",ms");
			}
		}
	}
	
	/**
	 * execute a runnable in the name of this threadpool
	 */
	protected void doIt(Runnable r) {
		long start = System.currentTimeMillis();
		if (logger.isLoggable(Level.FINER)) {
			logger.finer("executing " + r);
		}
		try {
			WorkUnit.work(r);
		} finally {
			synchronized (this) {
				this.tasksCompleted++;
			}
			if (perfLogger.isLoggable(Level.FINE)) {
				long durationTotal = System.currentTimeMillis() - start;
				perfLogger.fine("system.work.threadpool." + id + ",total work unit," + r + ",duration," + durationTotal + ",ms");
				perfLogger.fine("system.work.threadpool." + id + ",total runnable," + r + ",duration," + durationTotal + ",ms");
			}
		}
	}
	
	/**
	 * A thread from some other pool may execute a runnable and still account
	 * for this thread pool (in particular will be associated with the right 
	 * WorkUnit handling.
	 * Sometimes it may be useful to not limit the overall
	 * concurrency to the systems max concurrency in this case. 
	 */
	public void executeAs(final Runnable runnable, boolean maxconcurrency) {
		try {
			executeAs(new Callable<Void>() {
				public Void call() throws Exception {
					runnable.run();
					return null;
				}
			},maxconcurrency);
		} catch (RuntimeException re) {
			throw re;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * A thread from some other pool may execute a runnable and still account
	 * for this thread pool (in particular will be associated with the right 
	 * WorkUnit handling.
	 * Sometimes it may be useful to not limit the overall
	 * concurrency to the systems max concurrency in this case. 
	 */
	public <T>  T executeAs(Callable<T> t, boolean maxconcurrency) throws Exception {
		if (isPoolThread(Thread.currentThread())) {
			// just go there - we don't count recursive association as increase of concurrency
			return doIt(t);
		} else {
			foreignPoolThreads.add(Thread.currentThread());
			try {
				QueuedExecutionGate qg=null;
				synchronized (this) {
					// if bounded and max concurrency reached, wait
					if (maxconcurrency && this.conc>=this.maxconc) {
						// prepare for queued blocking
						qg = new QueuedExecutionGate();
						_dispatch(qg);
					} else {
						// else account concurrency right away
						_increaseConc();
					}
				}
				if (qg==null) {
					// do it inline - no fuss
					try {
						return doIt(t);
					} finally {
						synchronized (this) {
							// unaccount
							this._decreaseConc();
							// make sure work is being dispatched and tasks don't starve
							this._pickUpWork();
						}
					}
				} else {
					// let it be mediated by the pool queue
					return qg.execute(t);
				}
			} finally {
				foreignPoolThreads.remove(Thread.currentThread());
			}
		}
	}

	
	protected String getId() {
		return this.id;
	}

}
