/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.launch;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.concurrent.Callable;

import javax.naming.InitialContext;
import javax.naming.NameNotFoundException;

import com.zfabrik.launch.impl.ProcessRunnerImpl;
import com.zfabrik.launch.impl.Z2CommandLine;

/**
 * Main class that performs a lookup for a z2 component
 * and passes all parameters to 
 * a main method of the lookup result.
 * <p>
 * The providing resource should satisfy a lookup for <code>java.lang.Class</code> by returning
 * a class object that has a main method (i.e. a <code>public static void main(String[])</code> method).
 * <p>
 * This class is the main class of the <code>z_embedded.jar</code>. I.e. given a component <code>x</code> and
 * supposing that the <code>Z2_HOME</code> environment variable is set, calling the main method of <code>x</code>
 * in an embedded z2 environment can be achieved by the following command line:
 * <pre>
 * java -DcomponentName=&lt;x&gt; -jar z_embedded.jar &lt;args&gt; 
 * </pre>
 * 
 * @author hb
 *
 */

public class MainRunner extends Z2CommandLine {

	public MainRunner() {}
	
	public static void main(final String[] args) throws Exception {
		new MainRunner(args);
	}

	public MainRunner(final String[] args) throws Exception {
		final String component = System.getProperty("componentName");
		if (component==null) {
			usage();
		}
		ProcessRunnerImpl.start();
		try {
			// encapsulate in work unit
			ProcessRunnerImpl.work(new Callable<Void>() {
				public Void call() throws Exception {
					Object o = null;
					try {
						o = new InitialContext().lookup("components:"+component+"?type=java.lang.Class");
					} catch (NameNotFoundException nnfe) {
						error("Component \""+component+"\" not found or not supporting java.lang.Class lookups (and hence not usable here)");
					}
					if (!(o instanceof Class)) {
						error("Component lookup for \""+component+"\" did not return a Class object but instead an instance of "+o.getClass());
					}
					Class<?> clz = (Class<?>) o;
					Method m = null;
					try {
						m = clz.getMethod("main", String[].class);
					} catch (NoSuchMethodException e) {
						error("Class \""+o.getClass().getName()+"\" has no main(String[]) method");
					}
					if (!Modifier.isStatic(m.getModifiers()) || !Modifier.isPublic(m.getModifiers())) {
						error("Class \""+o.getClass().getName()+"\" has no static public main(String[]) method");
					}
					m.invoke(null,new Object[]{args});
					return null;
				}
			});
		}  finally {
			ProcessRunnerImpl.stop();
		}

		
	}

	private void error(String err) {
		System.err.println("MainRunner Error: "+err);
		usage();
	}
	

}
