/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.launch;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;

import com.zfabrik.launch.impl.Z2CommandLine;
import com.zfabrik.launch.util.LogStreamer;
import com.zfabrik.util.function.ThrowingRunnable;

/**
 * JMX based management utility
 *
 * @author Henning Blohm
 */
public class Manage extends Z2CommandLine {
	private static final String JMXRMI = "/jmxrmi";
	private static final String SERVICE_JMX_RMI_JNDI_RMI = "service:jmx:rmi:///jndi/rmi://";
	private final static String COMMAND_SYNC 	= "sync";
	private final static String COMMAND_SHOWLOG = "showlog";
	private final static String[] COMMANDS = { COMMAND_SHOWLOG, COMMAND_SYNC };
	
	private final static String OPTION_URL = "-url";
	private final static String OPTION_USER = "-user";
	private final static String OPTION_PASSWORD = "-pass";
	private final static String OPTION_BEFORE = "-b";
	private final static String[] OPTIONS = { OPTION_PASSWORD, OPTION_URL, OPTION_USER, OPTION_BEFORE };
	
	public static final Logger LOG = Logger.getLogger(Manage.class.getName());

	public Manage() {
	}
	
	public Manage(String[] args) throws Exception {
		super(args);
	}

	public static void main(String[] args) throws Exception {
		new Manage(args).run();
	}

	private void run() throws Exception {
		
		String url = getOptions().remove(OPTION_URL);
		String user= getOptions().remove(OPTION_USER);
		String pass= getOptions().remove(OPTION_PASSWORD);
		// either both or none:
		if ( (user==null) != (pass==null) ) {
			usage("Either specify user and pass or none");
		}
		// remove the options from the params 
		for (String c : OPTIONS) { getParams().remove(c); }
		// there should be one left: The command
		if (getParams().size()!=1) {
			usage("Found more than one command: "+getParams().keySet());
		}
		
		// check command
		String command = getParams().keySet().iterator().next();
		if (!Arrays.asList(COMMANDS).contains(command)) {
			usage("Unknown command "+command);
		}
		
		// defaults:
		if (url==null) {
			url = "localhost:7777";
		}
		// complete url
		if (!url.startsWith(SERVICE_JMX_RMI_JNDI_RMI)) {
			url = SERVICE_JMX_RMI_JNDI_RMI+url+JMXRMI;
		}
		
		
		// set up the JMX stuff
		Map<String,String[]> env = new HashMap<>();
		if (user!=null) {
			env.put(JMXConnector.CREDENTIALS, new String[]{user,pass});
		}

		JMXServiceURL jmx = new JMXServiceURL(url);
		MBeanServerConnection mbs = JMXConnectorFactory.connect(jmx).getMBeanServerConnection();

		// we have a connection, now do what we are asked for
		
		// set up the log stream
		switch (command) {
		case COMMAND_SYNC:
			new LogStreamer(mbs, 0).logWhile(async(sync(mbs)));
			break;
		case COMMAND_SHOWLOG:
			int n = Integer.parseInt(getOptions().getOrDefault(OPTION_BEFORE, "0"));
			// just keep logging
			new LogStreamer(mbs, n).logWhile(new AtomicBoolean(true));
			break;
		}
	}

	/**
	 * Perform some work in a new thread and return an {@link AtomicBoolean}
	 * indicating whether work is still on the way
	 */
	private AtomicBoolean async(ThrowingRunnable<Exception> work) {
		AtomicBoolean busy = new AtomicBoolean(true);
		new Thread(() -> {
			try {
				work.run();
			} catch (Throwable t) {
				if (t instanceof VirtualMachineError) {
					t.printStackTrace();
					System.exit(-1);
				}
				LOG.log(Level.SEVERE, "Error during command execution", t);
			} finally {
				// make we stop streaming the log!
				busy.set(false);
			}
		}).start();
		return busy;
	}

	/**
	 * A sync
	 */
	private ThrowingRunnable<Exception> sync(MBeanServerConnection mbs) {
		return ()->{
			mbs.invoke(ObjectName.getInstance("zfabrik:type=launchers.HomeLauncher"), "synchronize", new Object[] {}, new String[] {});
		};
	}

}
