/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.util.expression;

import java.util.Collection;
import java.util.Map;
import java.util.Properties;

/**
 * Simple boolean expression language as a Java expressed DSL. <code>X</code> is used in
 * component repositories to formulate component queries but may be used anywhere where 
 * simple boolean expressions are useful.
 * <p>
 * A built-in evaluation of an expression over a map defining a context is supported. 
 * <p>
 * For example, given a map <code>m={"a":5, "b":"hello"}</code>, 
 * <pre>
 * var("a").ge(val(6)).and(var("b").eq(val("hello"))).eval(m) 
 * </pre>
 * would evaluate to <code>false</code> while 
 * <pre>
 * var("a").le(val(6)).and(var("b").eq(val("hello"))).eval(m) 
 * </pre>
 * would evaluate to <code>true</code>.
 * 
 * @author hb
 *
 */

public abstract class X {
	
	/**
	 * Abstract binary operators
	 */
	public abstract static class BinaryOp extends X {
		private X x,y;
		
		public BinaryOp(X x, X y) {
			this.x = x;
			this.y = y;
		}
		
		/**
		 * return first or "left" operand expression
		 */
		public X getLeft() {return this.x;}
		/**
		 * return second or "right" operand expression
		 */
		public X getRight() {return this.y;}
		
		public boolean equals(Object obj) {
			if (obj==null) return false;
			if (obj==this) return true;
			if (obj.getClass().equals(this.getClass())) {
				BinaryOp that = (BinaryOp) obj;
				if ((x==null && that.x==null) || (x!=null && x.equals(that.x))) {
					if ((y==null && that.y==null) || (y!=null && y.equals(that.y))) {
						return true;
					}
				}
			}
			return false;
		}

		public int hashCode() {
			int s =   this.getClass().hashCode();
			if (x!=null) s ^= x.hashCode();
			if (y!=null) s ^= y.hashCode();
			return s;
		}		
	}
	
	/**
	 * "OR" logical operator
	 */
	public static class Or extends BinaryOp {
		public Or(X x,X y) { super(x,y); }

		public Object eval(Map<String, Object> context) {
			Object o1 = getLeft().eval(context);
			if (o1 instanceof Boolean) {
				if (((Boolean) o1).booleanValue()) 
					return true;
				else {
					Object o2 = getRight().eval(context);
					if (o2 instanceof Boolean) {
						return (Boolean) o2;
					} else {
						throw new IllegalArgumentException("expected boolean, found:" +o2+" when evaluating "+getRight());
					}
				}
			} else {
				throw new IllegalArgumentException("expected boolean, found:" +o1+" when evaluating "+getLeft());
			}
		}
		
		public String toString() {
			return "(or "+getLeft()+" "+ getRight()+")";
		}
	}
	
	/**
	 * "AND" logical operator
	 */
	public static class And extends BinaryOp {
		public And(X x,X y) { super(x,y); }

		public Object eval(Map<String, Object> context) {
			Object o1 = getLeft().eval(context);
			if (o1 instanceof Boolean) {
				if (!((Boolean) o1).booleanValue()) 
					return false;
				else {
					Object o2 = getRight().eval(context);
					if (o2 instanceof Boolean) {
						return (Boolean) o2;
					} else {
						throw new IllegalArgumentException("expected boolean, found:" +o2+" when evaluating "+getRight());
					}
				}
			} else {
				throw new IllegalArgumentException("expected boolean, found:" +o1+" when evaluating "+getLeft());
			}
		}

		public String toString() {
			return "(and "+getLeft()+" "+ getRight()+")";
		}
	}
	
	/**
	 * "XOR" logical operator
	 */
	public static class Xor extends BinaryOp {
		public Xor(X x,X y) { super(x,y); }

		public Object eval(Map<String, Object> context) {
			Object o1 = getLeft().eval(context);
			if (o1 instanceof Boolean) {
				Object o2 = getRight().eval(context);
				if (o2 instanceof Boolean) {
					return (((Boolean) o1).booleanValue() ^ ((Boolean) o2).booleanValue());
				} else {
					throw new IllegalArgumentException("expected boolean, found:" +o2+" when evaluating "+getRight());
				}
			} else {
				throw new IllegalArgumentException("expected boolean, found:" +o1+" when evaluating "+getLeft());
			}
		}

		public String toString() {
			return "(xor "+getLeft()+" "+ getRight()+")";
		}
	}

	private static int _compareNumbers(Object o1, Object o2) {
		if (o1==null) {
			throw new NullPointerException("Expected non-null left side of comparision");
		}
		if (o2==null) {
			throw new NullPointerException("Expected non-null right side of comparision");
		}
		if (o1 instanceof Number) {
			if (o2 instanceof Number) {
				Number n1 = Number.class.cast(o1);
				Number n2 = Number.class.cast(o2);
				if (n1.doubleValue()<n2.doubleValue()) {
					return -1;
				}
				if (n1.doubleValue()==n2.doubleValue()) {
					return 0;
				}
				return +1;
			} else {
				throw new IllegalArgumentException("Expected Number on right side but got "+o2.getClass().getName());
			}
		} else {
			throw new IllegalArgumentException("Expected Number on left side but got "+o1.getClass().getName());
		}
	}

	/**
	 * "less than" comparison operator.
	 */
	public static class Lt extends BinaryOp {
		public Lt(X x,X y) { super(x,y); }

		public Object eval(Map<String, Object> context) {
			return _compareNumbers(getLeft().eval(context),getRight().eval(context))<0;
		}

		public String toString() {
			return "(lt "+getLeft()+" "+ getRight()+")";
		}
	}

	/**
	 * "less or equal" comparison operator.
	 */
	public static class Le extends BinaryOp {
		public Le(X x,X y) { super(x,y); }

		public Object eval(Map<String, Object> context) {
			return _compareNumbers(getLeft().eval(context),getRight().eval(context))<=0;
		}

		public String toString() {
			return "(le "+getLeft()+" "+ getRight()+")";
		}
	}

	/**
	 * "greater than" comparison operator.
	 */
	public static class Gt extends BinaryOp {
		public Gt(X x,X y) { super(x,y); }

		public Object eval(Map<String, Object> context) {
			return _compareNumbers(getLeft().eval(context),getRight().eval(context))>0;
		}

		public String toString() {
			return "(gt "+getLeft()+" "+ getRight()+")";
		}
	}

	/**
	 * "greater or equal" comparison operator.
	 */
	public static class Ge extends BinaryOp {
		public Ge(X x,X y) { super(x,y); }

		public Object eval(Map<String, Object> context) {
			return _compareNumbers(getLeft().eval(context),getRight().eval(context))>=0;
		}

		public String toString() {
			return "(ge "+getLeft()+" "+ getRight()+")";
		}
	}

	/**
	 * "equality" comparison operator.
	 */
	public static class Eq extends BinaryOp {
		public Eq(X x,X y) { super(x,y); }

		public Object eval(Map<String, Object> context) {
			Object o1 = getLeft().eval(context);
			Object o2 = getRight().eval(context);
			if (o1!=null) {
				return o1.equals(o2);
			} else {
				return (o2==null);
			}
		}

		public String toString() {
			return "(eq "+getLeft()+" "+ getRight()+")";
		}
	}

	/**
	 * "not equals" comparison operator.
	 */
	public static class Neq extends BinaryOp {
		public Neq(X x,X y) { super(x,y); }

		public Object eval(Map<String, Object> context) {
			Object o1 = getLeft().eval(context);
			Object o2 = getRight().eval(context);
			if (o1!=null) {
				return !o1.equals(o2);
			} else {
				return (o2!=null);
			}
		}

		public String toString() {
			return "(neq "+getLeft()+" "+ getRight()+")";
		}

	}
	
	/**
	 * "containment" operator. Evaluates to <code>true</code>, if the left operand is contained 
	 * in the right operand - whatever that may mean in the evaluation context.
	 * <p>
	 * The default evaluation over a map context (see {@link #eval(Map)})) translates to collection containment
	 * if the right operand evaluates to a collection or to containment as showing up as a string
	 * in a comma-separated list of strings that is the right operand's evaluation result.  
	 */
	public static class In extends BinaryOp {
		public In(X x,X y) { super(x,y); }

		@SuppressWarnings("rawtypes")
		public Object eval(Map<String, Object> context) {
			Object o1 = getLeft().eval(context);
			Object o2 = getRight().eval(context);
			if (o2==null) {
				return false; // treat as empty list
			} else 
			if (o2 instanceof Collection) {
				return ((Collection) o2).contains(o1);
			} else if (o2 instanceof String) {
				// we accept comma-separated string lists.
				if (o1 instanceof String) {
					String s2 = (String) o2;
					String s1 = (String) o1;
					int p = s2.indexOf(s1);
					if (p>=0) {
						int a = p-1;
						int e = p+s1.length();
						while ((a>=0) && (Character.isWhitespace(s2.charAt(a)))) a--;
						while ((e<s2.length()) && (Character.isWhitespace(s2.charAt(e)))) e++;
						return ((a<0) || (s2.charAt(a)==',')) && ((e>=s2.length()) || (s2.charAt(e)==','));
					}
					return false;
				}
				throw new IllegalArgumentException("expected string, found:" +o1+" when evaluating "+getLeft());
			} else {
				throw new IllegalArgumentException("expected collection, found:" +o2+" when evaluating "+getRight());
			}
		}

		public String toString() {
			return "(in "+getLeft()+" "+ getRight()+")";
		}

	}

	
	/**
	 * Abstract single-operand operator
	 */
	public static abstract class UnaryOp extends X {
		private X x;
		public UnaryOp(X x) { this.x = x; }
		public X get() {return this.x;}

		public boolean equals(Object obj) {
			if (obj==null) return false;
			if (obj==this) return true;
			if (obj.getClass().equals(this.getClass())) {
				UnaryOp that = (UnaryOp) obj;
				if ((x==null && that.x==null) || (x!=null && x.equals(that.x))) {
					return true;
				}
			}
			return false;
		}
		public int hashCode() {
			if (x!=null) return this.getClass().hashCode() ^ x.hashCode();
			return 0;
		}		
	}

	/**
	 * Negation logical operator 
	 */
	public static class Not extends UnaryOp {
		public Not(X x) { super(x); }

		public Object eval(Map<String, Object> context) {
			Object o1 = get().eval(context);
			if (o1 instanceof Boolean) {
				return Boolean.valueOf(!((Boolean) o1).booleanValue()); 
			} else {
				throw new IllegalArgumentException("expected boolean, found:" +o1+" when evaluating "+get());
			}
		}

		public String toString() {
			return "(not "+get()+")";
		}
	}

	/**
	 * Variable of field value operator. The value of the {@link Var} operator is to 
	 * denote a variable or field name. During evaluation this name should be resolved
	 * to a value. 
	 * <p>
	 * In the default map context evaluation, the map value for the variable name is returned
	 * as evaluation result
	 * 
	 * @author hb
	 *
	 */
	public static class Var extends X {
		private String name;
		public Var(String name) { this.name = name; }
		public String get() { return this.name; }

		public Object eval(Map<String, Object> context) {
			return context.get(name);
		}

		public String toString() {
			return name;
		}

		public boolean equals(Object obj) {
			if (obj==null) return false;
			if (obj==this) return true;
			if (obj.getClass().equals(this.getClass())) {
				Var that = (Var) obj;
				if ((name==null && that.name ==null) || (name!=null && name.equals(that.name))) {
					return true;
				}
			}
			return false;
		}
		public int hashCode() {
			if (name!=null) return name.hashCode();
			return 0;
		}		
	}
	
	/**
	 * Explicit value. The value passed in to the {@link Var} operator is the value returned
	 * during evaluation.
	 *  
	 * @author hb
	 *
	 */
	public static class Val extends X {
		private Object val;
		public Val(Object val) { this.val = val; }
		public Object get() { return this.val; }

		public Object eval(Map<String, Object> context) {
			return val;
		}

		public String toString() {
			if (val instanceof String) {
				// TODO escape string
				return "\""+val.toString()+"\"";
			} else {
				return "["+val.toString()+"]";
			}
		}
		public boolean equals(Object obj) {
			if (obj==null) return false;
			if (obj==this) return true;
			if (obj.getClass().equals(this.getClass())) {
				Val that = (Val) obj;
				if ((val==null && that.val ==null) || (val!=null && val.equals(that.val))) {
					return true;
				}
			}
			return false;
		}
		public int hashCode() {
			if (val!=null)  return val.hashCode();
			return 0;
		}		
	}
	

	/**
	 * Convenience method for chain-style notation
	 * of expressions. 
	 */
	public X or(X x) { return new Or(this,x); }
	/**
	 * Convenience method for chain-style notation
	 * of expressions. 
	 */
	public X and(X x) { return new And(this,x); }
	/**
	 * Convenience method for chain-style notation
	 * of expressions. 
	 */
	public X xor(X x) { return new Xor(this,x); }
	/**
	 * Convenience method for chain-style notation
	 * of expressions. 
	 */
	public X eq(X x) { return new Eq(this,x); }
	/**
	 * Convenience method for chain-style notation
	 * of expressions. 
	 */
	public X lt(X x) { return new Lt(this,x); }
	/**
	 * Convenience method for chain-style notation
	 * of expressions. 
	 */
	public X le(X x) { return new Le(this,x); }
	/**
	 * Convenience method for chain-style notation
	 * of expressions. 
	 */
	public X gt(X x) { return new Gt(this,x); }
	/**
	 * Convenience method for chain-style notation
	 * of expressions. 
	 */
	public X ge(X x) { return new Ge(this,x); }
	/**
	 * Convenience method for chain-style notation
	 * of expressions. 
	 */
	public X in(X x) { return new In(this,x); }
	/**
	 * Convenience method for chain-style notation
	 * of expressions. 
	 */
	public X neq(X x) { return new Neq(this,x); }
	/**
	 * Convenience method for chain-style notation
	 * of expressions. 
	 */
	public X not() { return new Not(this); }

	/**
	 * Static factory method, for convenience
	 */
	public static X or(X x, X y) { return new Or(x,y); };
	/**
	 * Static factory method, for convenience
	 */
	public static X and(X x, X y) { return new And(x,y); };
	/**
	 * Static factory method, for convenience
	 */
	public static X xor(X x, X y) { return new Xor(x,y); };
	/**
	 * Static factory method, for convenience
	 */
	public static X eq(X x, X y) { return new Eq(x,y); };
	/**
	 * Static factory method, for convenience
	 */
	public static X neq(X x, X y) { return new Neq(x,y); };
	/**
	 * Static factory method, for convenience
	 */
	public static X not(X x) { return new Not(x); };
	/**
	 * Static factory method, for convenience
	 */
	public static X val(Object val) { return new Val(val); };
	/**
	 * Static factory method, for convenience
	 */
	public static X var(String name) { return new Var(name); };
	
	/**
	 * Evaluation of the expression over a map style context that 
	 * defines variable values.
	 */
	public abstract Object eval(Map<String,Object> context);

	/**
	 * Convenience version of {@link #eval(Map)} accepting a {@link Properties} argument
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public Object eval(Properties context) {
		return this.eval((Map) context);
	}

	/**
	 * Convenience version of {@link #eval(Map)} that is equivalent to calling {@link #eval(Map)} with an 
	 * empty map or a null argument.
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	public Object eval() {
		return this.eval((Map) null);
	}
	
}
