/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.work;

import java.util.Collection;
import java.util.concurrent.Callable;

/**
 * thread pool abstraction provided by the work manager. Threadpools are intimately related to work units in that the threads of one pool
 * define outer boundaries for work units.  Nested {@link #executeAs(Runnable, boolean)} executions on the same pool
 * will share one work unit, while switching from one 
 * pool to another will lead to independent work units.
 * Threads not owned by this pool may be joined into this pool (and hence the workunit mechanism and the max concurrency config) via the executeAs method. 
 */
public interface IThreadPool {
    /**
     * Execute the passed on tasks concurrently within the bounds of the pools configured 
     * maximal concurrency.
     * <p>
     * In case a non-blocking execution is required, all tasks will be put into the threadpool's task queue. In extreme cases this may imply
     * that no task will be started before the current thread returns to the pool.
     * <br/> 
     * In particular, if the calling thread is from the same pool, it is crucial to not have the calling thread wait
     * for the completion of the tasks, as those may never get started as all the pool's threads are busy. If a number of tasks need to be 
     * completed and the current thread must not return to the pool before completion, this method should be called indicating a blocking
     * execution 
     * </p>
     * 
     * @param runnables an array of runnables to be executed.
     * @param block if <code>true</code> the current thread will not return until all runnables have executed. If <code>false</code>, the 
     * current thread will return immediately.
     * @throws WorkException if there occurred an uncaught and non-severe exception during 
     * execution of a runnable. Any runnables that has not been started yet will not be started at all.  
     */
    void execute(boolean block, Collection<? extends Runnable> runnables);
    
    /**
     * short hand method.
     * @param runnables
     * @param block if <code>true</code> the current thread will not return until all runnables have executed. If <code>false</code>, the 
     * current thread will not return if it is from the same thread pool and will return otherwise.
     */
    void execute(boolean block, Runnable ... runnables);
    
    /**
     * sets the max concurrency for this thread pool. This number of threads is guaranteed to be made available
     * if needed. Any more active jobs will be queued
     */
    void setMaxConcurrency(int maxconcurrency);
    
    /**
     * sets the max concurrency for this thread pool. This number of threads is guaranteed to be made available
     * if needed. Any more active jobs will be queued
     */
    int getMaxConcurrency();

    /**
     * get the max achieved concurrency
     */
    int getMaxAchievedConcurrency();
    
    /**
     * checks whether a given thread is of this pool
     */
    boolean isPoolThread(Thread t);
    
    /**
     * Incorporate the current thread logically into the pool, if it is not one of the pool's threads and execute the runnable. While this may lead to an effective increase of overall concurrency, some frameworks
     * leave no choice between risking deadlocks when switching threads or not being able to attach a suitable thread context when not switching threads (e.g. session invalidation in Jetty)...
     * If maxconcurrency is set to true, the calling thread may need to wait in order to not exceed our configured max concurrency
     */
    void executeAs(Runnable runnable, boolean maxconcurrency);

    /**
     * Like above, but excepting a callable and thereby providing for an exception flow and a return value 
     */
    <T>  T executeAs(Callable<T> runnable, boolean maxconcurrency) throws Exception;
    
}
