/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.any;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.components.IDependencyComponent;
import com.zfabrik.components.java.JavaComponentUtil;
import com.zfabrik.resources.ResourceBusyException;
import com.zfabrik.resources.provider.Resource;


/**
 * An arbitrary resource. For any class, a resource will be maintained that
 * returns for any implemented class or interface of the given class.
 * 
 * If the class implements Resource, all Resource calls will be delegated.
 * 
 * If the class implements Runnable, it will be called upon system state attain operations.
 * 
 */
public class AnyResource extends Resource {
	private AR ar;
	private String name;
	
	public AnyResource(String name) {
		this.name = name;
	}
	
	private static class AR {
		@SuppressWarnings("unused")
		private AnyResource ar;
		private Resource re;
		
		public AR(AnyResource ar, Resource re) {
			super();
			this.ar = ar;
			this.re = re;
		}

		public <T> T as(Class<T> clz) {
			if (IDependencyComponent.class.equals(clz)) {
				IDependencyComponent sl = re.as(IDependencyComponent.class);
				if (sl==null) {
					// try as runnable (either provided or castable)
					Runnable rx = re.as(Runnable.class);
					if (rx==null && (re instanceof Runnable)) {
						rx = (Runnable) re;
					}
					final Runnable ru = rx;
					if (ru!=null) {
						sl = new IDependencyComponent() {
							public void prepare() {
								ru.run();
							}
						};
					}
				}
				return clz.cast(sl);
			}
			return re.as(clz);
		}
		
		public void invalidate() {
			this.re.invalidate();
		}
	}

	public <T> T as(Class<T> clz) {
		synchronized (this) {
			if (this.ar==null) {
				IComponentDescriptor d = IComponentsManager.INSTANCE.getComponent(this.name);
				String cln = d.getProperty(IComponentDescriptor.COMPONENT_CLZ);
				Object y;
				if (cln==null) {
					// "any" component do not require a class name. If none is present, they
					// default to "java.lang.Object"
					y = new Object();
				} else {
					Class<?> c = JavaComponentUtil.loadImplementationClassFromJavaComponent(
						this.name, 
						IComponentDescriptor.COMPONENT_CLZ, 
						handle()
					);
					try {
						if (Resource.class.isAssignableFrom(c)) {
							// In case of Resource, support string arg constructor (#2061)
							try {
								y = c.getConstructor(String.class).newInstance(this.name);
							} catch (NoSuchMethodException nsme) {
								// try with default constructor
								y = c.getConstructor().newInstance();
							}
						} else {
							// try with default constructor
							y = c.getConstructor().newInstance();
						}
					} catch (LinkageError le) {
						throw new IllegalStateException("failed to load implementation of component: "+this.name,le);
					} catch (Exception e) {
						throw new IllegalStateException("failed to load implementation of component: "+this.name,e);
					}
				}
				final Object x = y;
				Resource r;
				if (x instanceof Resource) {
					r = (Resource) x;
				} else {
					// create a simple resource that returns the original object 
					// we don't have any but Java component dependencies, which is
					// why we do not need to explicitly invalidate here but rely
					// on the component resource wrapper to dump the resource. 
					r = new Resource() {
						public void init() {
							// attach the handle to x.
							handle().attach(x);
						}
						public <S> S as(Class<S> clz) {
							if (clz!=null && clz.isAssignableFrom(x.getClass())) {
								return clz.cast(x);
							}
							return super.as(clz);
						}
					}; 
				}
				r.init(handle());
				this.ar = new AR(this,r);
			}
			return this.ar.as(clz);
		}
	}

	public void invalidate() throws ResourceBusyException {
		synchronized (this) {
			if (this.ar!=null) {
				this.ar.invalidate();
			}
		}
		
	}

}
