/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.java;

import static com.zfabrik.components.java.IJavaComponent.COMPILE_ORDER;
import static com.zfabrik.components.java.IJavaComponent.DEFAULT_CLASSPATH_PATTERN;
import static com.zfabrik.components.java.IJavaComponent.PRIINCS;
import static com.zfabrik.components.java.IJavaComponent.PRIREFS;
import static com.zfabrik.components.java.IJavaComponent.PRIVATE_CLASSPATH_PATTERN;
import static com.zfabrik.components.java.IJavaComponent.PUBINCS;
import static com.zfabrik.components.java.IJavaComponent.PUBLIC_CLASSPATH_PATTERN;
import static com.zfabrik.components.java.IJavaComponent.PUBREFS;
import static com.zfabrik.components.java.IJavaComponent.TESTINCS;
import static com.zfabrik.components.java.IJavaComponent.TESTREFS;
import static com.zfabrik.components.java.IJavaComponent.TEST_CLASSPATH_PATTERN;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.components.java.IJavaComponent;
import com.zfabrik.components.java.IJavaComponent.Part;
import com.zfabrik.components.java.JavaComponentUtil;
import com.zfabrik.components.java.LangLevel;
import com.zfabrik.components.java.build.ICompiler;
import com.zfabrik.components.java.build.IJavaBuilder;
import com.zfabrik.components.provider.util.LockingRevFile;
import com.zfabrik.impl.compiler.AllCompiler;
import com.zfabrik.resources.IResourceHandle;
import com.zfabrik.resources.ResourceBusyException;
import com.zfabrik.resources.provider.Resource;
import com.zfabrik.util.fs.FileUtils;
import com.zfabrik.util.runtime.Foundation;


/**
 * This class implements the actual make for Java components. See {@link IJavaComponent} for the 
 * supported folder structure.
 * 
 * @author hb
 *
 */
public class ComponentsBuilder extends Resource implements IJavaBuilder {

	public <T> T as(Class<T> clz) {
		if (IJavaBuilder.class.equals(clz)) {
			return clz.cast(this);
		}
		return null;
	}

	public void invalidate() throws ResourceBusyException {}
	
	// source folders for java components
	private final static String SRC_API = "src.api";
	private final static String SRC_IMPL = "src.impl";
	private final static String SRC_TEST = "src.test";

	// bin folders for java components
	private static final String BIN_API = "bin.api";
	private static final String BIN_IMPL = "bin.impl";
	private static final String BIN_TEST = "bin.test";
	private static final String BIN_API_CLASSES = BIN_API+"/classes";
	private static final String BIN_API_LIB = BIN_API+"/lib";
	private static final String BIN_IMPL_CLASSES = BIN_IMPL+"/classes";
	private static final String BIN_IMPL_LIB = BIN_IMPL+"/lib";
	private static final String BIN_TEST_CLASSES = BIN_TEST+"/classes";
	private static final String BIN_TEST_LIB = BIN_TEST+"/lib";

	// alternatives to *.api
	private static final String SRC = "src";
	private static final String BIN_LIB = "bin/lib";
	private static final String BIN_CLASSES = "bin/classes";

	// deprecated:
	private static final String BIN_PRIVATE_CLASSES = "bin/private/classes";
	private static final String BIN_PRIVATE_LIB = "bin/private/lib";

	// source folders for includes (file components)
	private static final String INCL_SRC = "src";
	// 
	private static final String INCL_BIN_LIB = "bin/lib";
	private static final String INCL_BIN_CLASSES = "bin/classes";

	// other supporting folders
	private final static String GEN = "gen";
	private final static String DEP_FILE = GEN+"/dependencies.map";
	private final static String CMP_PRFX = "cmp.";
	private final static String PROP_PRFX = "prop.";
	private final static String TSTAMP_FILE = GEN+"/tstamp";
	private static final String TSTAMP = "lastBuild";
	
	
	private ICompiler compiler = new AllCompiler();
	private boolean   offline  = false;
	
	// strictly for unit testing
	public void setCompiler(ICompiler compiler) {
		this.compiler = compiler;
	}
	
	// strictly for unit testing
	public void setOffline(boolean offline) {
		this.offline = offline;
	}
	
	/**
	 * Note: we synchronize here, as we have to traverse the component graph and use file locks
	 * to avoid cross-process overwrites.
	 * Concurrent make would lead to duplicate file lock acquisition attempts and hence 
	 * Exceptions (besides the fact that multiple compilers are a resource risk).
	 */
	public synchronized void make(IComponentDescriptor cd, File originFolder, File instanceFolder,	Logger exLogger) {
		try {
			File lff = _getRevFile(instanceFolder);
			if (!lff.exists()) {
				lff.getParentFile().mkdirs();
			}
			LockingRevFile lockFile = new LockingRevFile(lff);
			lockFile.open();
			try {
				logger.finer("Inspecting component for make: "+cd.getName());
				/*
				 *  Check whether a build is needed.
				 *  
				 *  Note, if the revision of this component changed, our build result will be gone. So we do not need to check for that 
				 *  
				 *  First check if the component was built using the current runtime mode (null or "development")
				 *  
				 *  Secondly check our dependencies and their build time stamp or repository revisions. 
				 *  If available we compare that to a dependency map of previous build timestamps and revisions.
				 *  
				 *  Note: For Java components we generally look at the build time stamp so that component
				 *  changes propagate up the reference graph (API changes require transitive rebuild).
				 *  For non-Java components (currently only compilers) we use the component revision as
				 *  dependency discriminant.
				 *  
				 *  If they match no build is necessary
				 */
				
				// get the run mode
				String currMode = System.getProperty(Foundation.MODE);
				boolean devMode = Foundation.MODE_DEVELOPMENT.equals(currMode);

				// get the lang level
				LangLevel currLevel = LangLevel.determine();
				if (logger.isLoggable(Level.FINER)) {
					logger.finer("Using language level "+currLevel+" for "+cd.getName());
				}
				// pub refs
				Collection<String> pubdeps = splitDepString(cd.getProperty(PUBREFS));
				// pub includes
				Collection<String> pubincs = splitDepString(cd.getProperty(PUBINCS));
				// priv refs		
				Collection<String> privdeps = splitDepString(cd.getProperty(PRIREFS));
				// priv includes
				Collection<String> privincs = splitDepString(cd.getProperty(PRIINCS));

				// test refs
				Collection<String> testdeps;
				Collection<String> testincs;
				if (devMode) {
					testdeps = splitDepString(cd.getProperty(TESTREFS));
					testincs = splitDepString(cd.getProperty(TESTINCS));
				} else {
					testdeps = Collections.emptySet();
					testincs = Collections.emptySet();
				}
				
				//
				// Compiler dependencies: 
				// We must take care to depend on the revision of the compiler component 
				// and on its implementation.
				// 
				Set<String> cp = new HashSet<String>(1);
				Properties currentCompilerRevs = new Properties(); 
				
				if (!offline) {
					String cps = cd.getProperties().getProperty(COMPILE_ORDER,"java");
					// collect deps for re-use later.
					StringTokenizer tk = new StringTokenizer(cps,",");
					
					while (tk.hasMoreTokens()) {
						// we compare with the java component of the compiler component
						String cmn = JavaComponentUtil.getCompilerComponentById(tk.nextToken().trim());
						cp.add(cmn);
						// memorize for dep map update
						currentCompilerRevs.setProperty(CMP_PRFX+cmn, Long.toString(IComponentsManager.INSTANCE.getRevision(cmn)));
						// add the java component as well
						String cpn = JavaComponentUtil.getJavaComponentName(cmn);
						cp.add(cpn);
						// force up-to-date resources
						IComponentsLookup.INSTANCE.lookup(cpn, IJavaComponent.class);
						// memorize for dep map update
						currentCompilerRevs.setProperty(CMP_PRFX+cpn, Long.toString(_getRev(cpn)));
					}
				}				
				
				boolean doAPI  = true;
				boolean doIMPL = true;
				boolean doTEST = devMode;
				//
				// check deps
				//
				
				
				File depFile = new File(instanceFolder,DEP_FILE);
				Properties depmap = new Properties();
				if (depFile.exists()) {
					// something is there. Check on a more fine-grained level
					try {
						FileInputStream in = new FileInputStream(depFile);
						try {
							depmap.load(in);
						} finally {
							in.close();
						}
						
						String usedMode = depmap.getProperty(PROP_PRFX+Foundation.MODE);
						boolean modeChanged = !Objects.equals(currMode, usedMode);
						LangLevel usedLevel = LangLevel.parse(depmap.getProperty(Foundation.LANGUAGE_LEVEL));
						boolean levelChanged = !Objects.equals(currLevel,usedLevel);
						boolean compilersChanged = _requiresMake(cp, depmap);
					
						if (compilersChanged) {
							logger.finer("Found compiler change for "+cd.getName());
						}
						if (modeChanged) {
							logger.finer("Found mode change for "+cd.getName()+". Previously "+usedMode+", now "+currMode);
						}
						if (levelChanged) {
							logger.finer("Found language level change for "+cd.getName()+". Previously "+usedLevel+", now "+currLevel);
						}
						
						doAPI = compilersChanged || levelChanged || modeChanged || _requiresMake(_union(pubdeps, pubincs),depmap);
						doIMPL = doAPI 	|| _requiresMake(_union(privdeps, privincs),	depmap);
						doTEST = devMode && (doAPI || doIMPL || _requiresMake(_union(testdeps, testincs), depmap));
								
					} catch (IOException e) {
						logger.warning("Error when trying to read dependency description. Will rebuild");
					} finally {
						depmap.clear();
					}						
				} 
				
				boolean doSome = doAPI || doIMPL || doTEST;
				
				if (doSome) {
					
					logger.fine("Some make required for component "+cd.getName()+": {api: "+doAPI+", impl: "+doIMPL+", test: "+doTEST+"}");
					String projName = cd.getName();
					int p = projName.lastIndexOf('/');
					if (p>=0) {
						projName = projName.substring(0,p);
					}
						
					// prepare instance based resource folder from original component
					_pullFromOriginalComponentResources(originFolder, instanceFolder);
						
					//
					//  API make
					// 
					File bin_classes = new File(instanceFolder,BIN_CLASSES);
					File bin_api_classes = new File(instanceFolder,BIN_API_CLASSES);
					File bin_lib = new File(instanceFolder,BIN_LIB);
					File bin_api_lib = new File(instanceFolder,BIN_API_LIB);
					File src_api = new File(instanceFolder,SRC_API);
					File src = new File(instanceFolder,SRC);
					File api_jar = new File(bin_api_lib,projName+".api.jar");
					
	
					// conditional classpath inclusion
					Pattern clpIncl = Pattern.compile(cd.getProperties().getProperty(PUBLIC_CLASSPATH_PATTERN,DEFAULT_CLASSPATH_PATTERN));

					ClassLoader apiLoader = _makePart(
						instanceFolder,
						IResourceHandle.class.getClassLoader(),
						clpIncl,
						pubdeps,
						pubincs,
						depmap, 
						cd.getName(), 
						new File[]{src_api,src},
						new File[]{bin_api_lib, bin_lib},
						new File[]{bin_api_classes,bin_classes},
						api_jar,
						cd.getName()+"/api",
						doAPI,
						Part.PUBLIC
					);
						
					//
					//  Impl make
					// 
					
					File bin_impl_lib = new File(instanceFolder,BIN_IMPL_LIB);
					File bin_impl_classes = new File(instanceFolder,BIN_IMPL_CLASSES);
					File bin_private_lib = new File(instanceFolder,BIN_PRIVATE_LIB);
					File bin_private_classes = new File(instanceFolder,BIN_PRIVATE_CLASSES);
					File src_impl = new File(instanceFolder,SRC_IMPL);
					File impl_jar = new File(bin_impl_lib,projName+".impl.jar");

					// conditional classpath inclusion
					clpIncl = Pattern.compile(cd.getProperties().getProperty(PRIVATE_CLASSPATH_PATTERN,DEFAULT_CLASSPATH_PATTERN));
					
					ClassLoader implLoader = _makePart( 
						instanceFolder,
						apiLoader, 
						clpIncl,
						privdeps,
						privincs,
						depmap, 
						cd.getName(), 
						new File[]{src_impl},
						new File[]{bin_impl_lib, bin_private_lib},
						new File[]{bin_impl_classes, bin_private_classes},
						impl_jar,
						cd.getName()+"/impl",
						doIMPL,
						Part.PRIVATE
					);						
	
					//
					//  test make
					// 
					File bin_test_lib = new File(instanceFolder,BIN_TEST_LIB);
					File bin_test_classes = new File(instanceFolder,BIN_TEST_CLASSES);
					File src_test = new File(instanceFolder,SRC_TEST);
					File test_jar = new File(bin_test_lib,projName+".test.jar");
					
					// conditional classpath inclusion
					clpIncl = Pattern.compile(cd.getProperties().getProperty(TEST_CLASSPATH_PATTERN,DEFAULT_CLASSPATH_PATTERN));

					_makePart( 
						instanceFolder,
						implLoader,
						clpIncl,
						testdeps,
						testincs,
						depmap, 
						cd.getName(), 
						new File[]{src_test},
						new File[]{bin_test_lib},
						new File[]{bin_test_classes},
						test_jar,
						cd.getName()+"/test",
						doTEST,
						Part.TEST
					);										
						
					// write dependencies map and timestamp
					try {
						FileOutputStream out = new FileOutputStream(depFile);
						// add mode
						if (currMode!=null) {
							depmap.put(PROP_PRFX+Foundation.MODE,currMode);
						}
						// add level
						depmap.put(Foundation.LANGUAGE_LEVEL, currLevel.toString());
						// add previously collected compiler deps
						depmap.putAll(currentCompilerRevs);
						try {depmap.store(out, null);} finally {out.close();	}
					} catch (IOException ioe) {
						throw new RuntimeException("Component build failed when writing dependency map: "+cd.getName(),ioe);
					}
					// update my build timestamp
					lockFile.properties().put(TSTAMP, Long.toString(System.currentTimeMillis()));
					lockFile.update();
				} else {
					logger.finer("No make required for component "+cd.getName()+": {api: "+doAPI+", impl: "+doIMPL+", test: "+doTEST+"}");
				}
			} finally {
				lockFile.close();
			}
		} catch (Exception e) {
			throw new RuntimeException("Local make for component failed: "+cd.getName(),e);
		}
	}

	/*
	 * Split a dependency string and return it as a set
	 */
	private Collection<String> splitDepString(String s) {
		return JavaComponentUtil.parseDependencies(s);
	}
	
	private Collection<String> _union(Collection<String> one, Collection<String> two) {
		Set<String> u = new HashSet<String>(one.size()+two.size());
		u.addAll(one);
		u.addAll(two);
		return u;
	}

	//
	// check dependencies and return whether a make is required
	//
	private boolean _requiresMake(Collection<String> deps, Properties depmap) throws IOException {
		for (String dc:deps) {
			String lt = (String) depmap.get(CMP_PRFX+dc);
			if (lt!=null) {
				long vl = Long.parseLong(lt);
				IComponentDescriptor d = IComponentsLookup.INSTANCE.lookup(dc, IComponentDescriptor.class);
				if (d!=null) {
					// component exists
					if (IJavaComponent.TYPE.equals(d.getType()) && !JavaComponentUtil.isPreBuilt(d)) {
						// for built java components we retrieve the build time stamp
						long vt = _getRev(dc);
						if (vl==vt) {
							// still ok, continue
							continue;
						}
						logger.finer("Make required due to Java dependency version mismatch: Currently based on rev "+lt+" but have "+vt+" of "+dc);
					} else {
						// for non-java components or no-build components we retrieve the component revision
						if (vl==d.getRevision()) {
							// still ok, continue
							continue;
						}
						logger.finer("Make required due to dependency version mismatch: Currently based on rev "+lt+" but have "+d.getRevision()+" of "+dc);
					}
				} else {
					logger.finer("Make required due to unresolvable dependency "+dc);
				}
			} else {
				logger.finer("Make required due to add dependency "+dc);
			}
			// some mismatch or non-existing dependency
			return true;
		}
		return false;
	}
	
	//
	// build aspect (api, impl, test). Returns class loader that provides access to the part, incl. the build result 
	//
	private ClassLoader _makePart(
			File instanceFolder,    // Component resources to work on 
			ClassLoader parent, 	 // parent clp
			Pattern clpIncl,		 // classpath inclusion
			Collection<String> refs, // component refs
			Collection<String> incs, // component includes
			Properties depmap,  	 // dependency map to update
			String component, 	     // the component
			File[] srcs, 			 // all src folders, the first one is the default
			File[] libs, 			 // all jar folders, the first one is the default
			File[] classess, 		 // all classes folders, the first one is the default
			File target, 			 // target jar file
			String title,			 // message title for logging
			boolean doBuild,		 // if true do actually build if required. Otherwise just compute clp.
			IJavaComponent.Part part // the actual part specification
	) throws Exception {
		
		if (doBuild) {
			// delete any possible left fragment from a previously failed build
			FileUtils.delete(target);
		}

		if (doBuild && logger.isLoggable(Level.FINE)) {
			logger.fine(
				"Running make\n"+
				"component: "+component+"\n"+
				"   parent: "+parent+"\n"+
				"  clpIncl: "+clpIncl+"\n"+
				"     refs: "+refs+"\n"+
				"     incs: "+incs+"\n"+
				"   depmap: "+depmap+"\n"+
				"     srcs: "+Arrays.asList(srcs)+"\n"+
				"     libs: "+Arrays.asList(libs)+"\n"+
				"  classes: "+Arrays.asList(classess)+"\n"+
				"   target: "+target+"\n"+
				"     part: "+part+"\n"
			);
		}

		// the first ones are the default folders
		File lib = libs[0];
		File classes = classess[0];
		File src = srcs[0];
		
		List<ClassLoader> parents = new LinkedList<ClassLoader>();
		parents.add(parent);
		
		if (!offline) {
			// handle all refs
			if (refs.size()>0) {
				for (String r:refs) {
					IJavaComponent jc = IComponentsLookup.INSTANCE.lookup(r, IJavaComponent.class);
					if (jc!=null) {
						parents.add(jc.getPublicLoader());
						// note down dependency revision
						depmap.setProperty(CMP_PRFX+r,Long.toString(_getRev(r)));
					} else {
						throw new IllegalStateException(title+": Failed to resolve Java component "+r+" while making component");
					}
				}
			}
			// handle includes before building
			if (incs.size()>0) {
				for (String r:incs) {
					boolean isJavaComp = false;
					File from = IComponentsLookup.INSTANCE.lookup(r, File.class);
					if (from==null) {
						// could be a java component
						IJavaComponent jc;
						if ((jc=IComponentsLookup.INSTANCE.lookup(r, IJavaComponent.class))!=null) {
							// we go by the instance specific and processed resources
							from = jc.getRuntimeResources();
							isJavaComp = true;
						}
					}
					if (from!=null) {
						//
						// ok, copy stuff over
						//
						File s = new File(from,INCL_BIN_LIB);
						if (s.exists()) {
							FileUtils.copy(s, lib, null);
						}
						s = new File(from,INCL_BIN_CLASSES);
						if (s.exists()) {
							FileUtils.copy(s, classes, null);
						}
						if (!isJavaComp) {
							// copy sources only if not java component (as it's prebuild anyway)
							s = new File(from,INCL_SRC);
							if (s.exists()) {
								FileUtils.copy(s, src, null);
							}
						}
						if (isJavaComp) {
							// also check for api folders
							s = new File(from,BIN_API_LIB);
							if (s.exists()) {
								FileUtils.copy(s, lib, null);
							}
							s = new File(from,BIN_API_CLASSES);
							if (s.exists()) {
								FileUtils.copy(s, classes, null);
							}
						}
						// dep map
						if (isJavaComp) {
							// for java components we use the build time as rev
							depmap.setProperty(CMP_PRFX+r,Long.toString(_getRev(r)));
						} else {
							// otherwise the repository rev
							depmap.setProperty(CMP_PRFX+r,Long.toString(IComponentsManager.INSTANCE.getRevision(r)));
						}
					} else {
						throw new IllegalStateException(title+": Failed to resolve include component "+r+" while making component");
					}
				}
			}
		}
		
		// construct build classpath
		List<URL> clp = new LinkedList<URL>();
		for (File g : libs) {
			if (g.exists() && g.isDirectory()) {
				for (File f: g.listFiles(f->!f.isDirectory() && clpIncl.matcher(f.getName()).matches())) {
					clp.add(f.toURI().toURL());
				}
			}
		}		
		for (File g : classess) {
			if (g.exists() && g.isDirectory()) {
				clp.add(g.toURI().toURL());
			}
		}
		if (logger.isLoggable(Level.FINER) && !clp.isEmpty()) {
			logger.finer("Local classpath for "+title+" is "+clp);
		}

		// class path loader
		ClassLoader clpLoader = new ComponentClassLoader(
				null, 
				title, 
				clp.toArray(new URL[clp.size()]), 
				parents.toArray(new ClassLoader[parents.size()])
		);
		
		// check wether the src folders do exists (i.e. is there anything to compile)
		List<File> allsrcs = new LinkedList<File>();
		for (File s : srcs) {
			if (s.exists()) {
				allsrcs.add(s);
			}
		}
		if (doBuild) {
			if (!allsrcs.isEmpty()) {
				// got something to do 
				logger.fine("Running make of "+title);
				long start = System.currentTimeMillis();
				//
				// construct class path for private compile
				// from the many... many... sources
				//
				// output
				File gen_classes = new File(instanceFolder,GEN+"/classes");
				FileUtils.delete(gen_classes);
				gen_classes.mkdirs();
				target.getParentFile().mkdirs();
				
				// and compile
				BuildHelper.compileAndJar(
						component, 
						instanceFolder,
						srcs, 
						gen_classes, 
						target,
						clpLoader,
						compiler,
						part
				);
				// cleanup
				BuildHelper.delete(gen_classes);
				long duration = System.currentTimeMillis()-start;
				if (logger.isLoggable(Level.FINE)) {
					logger.fine("Make of "+title+" completed after "+duration+"ms");
				}
				// construct loader incl build result
				return new ComponentClassLoader(null, title+"/complete", new URL[]{target.toURI().toURL()}, new ClassLoader[]{clpLoader});
			} else {
				logger.fine("Nothing to compile for "+title);
			}
		}
		return clpLoader;
	}
	
	//
	// pull content back from cache, if we copied it there before. Need to make sure to not delete 
	// too much - i.e. build results from previously that we are not going to rebuild....
	//
	private void _pullFromOriginalComponentResources(File originFolder, File instanceFolder) throws Exception {
		FileUtils.copy(originFolder, instanceFolder, null);
	}

	private File _getRevFile(File componentFolder) throws IOException {
		File r = new File(JavaComponentUtil.getInstanceFolder(componentFolder),TSTAMP_FILE);
		return r;
	}
	
	private long _getRev(String component) throws IOException {
		File f = _getRevFile(IComponentsManager.INSTANCE.retrieve(component));
		if (!f.exists()) {
			// not yet built
			return 0;
		}
		LockingRevFile lock = new LockingRevFile(f);
		lock.open();
		try {
			return Long.parseLong(lock.properties().getProperty(TSTAMP));
		} finally {
			lock.close();
		}
	}
	
	protected final static Logger logger = Logger.getLogger(ComponentsBuilder.class.getName());
	
}
