/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.java.jdt;
import java.io.File;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.logging.Logger;

import org.eclipse.jdt.internal.compiler.classfmt.ClassFileReader;
import org.eclipse.jdt.internal.compiler.classfmt.ClassFormatException;
import org.eclipse.jdt.internal.compiler.env.INameEnvironment;
import org.eclipse.jdt.internal.compiler.env.NameEnvironmentAnswer;

import com.zfabrik.components.java.IJavaComponentClassLoader;

/**
 * Input to the jdt compiler. A name env provides classes to link to, sources to compile...
 */
public class NameEnvironmentImpl implements INameEnvironment {
	private final static Logger LOG = Logger.getLogger(NameEnvironmentImpl.class.getName());
	private ClassLoader cl;
	private File[] sourceFolders;
	private Map<String,File> sfnToFile = new HashMap<String, File>();
	private String encoding;
	private Map<String, Boolean> packages = new HashMap<>();

	public NameEnvironmentImpl(ClassLoader cl, String encoding, File... sourceFolders) {
		this.cl = cl;
		this.sourceFolders = sourceFolders;
		this.encoding = encoding;
	}

	public void cleanup() {
		this.sfnToFile.clear();
	}

	public NameEnvironmentAnswer findType(char[][] compoundName) {
		return _findType(concat(compoundName,'/'));
	}

	public NameEnvironmentAnswer findType(char[] typeName, char[][] packageName) {
		return _findType(getName(packageName, typeName, '/'));
	}

	/*
	 * The eclipse compiler checks, if a name (derived from using a symbol in code) is a package.
	 * This can be unexpected stuff like java.MyClass, mypackage.String.
	 * 
	 * On the other hand, if we claim java.MyClass IS a package, it may stop looking deeper for
	 * MyClass (which is actually in the same package as the referencing class).
	 * 
	 * Hence we may not easily claim something is a package, if it actually is a class.
	 *
	 * And then, the system and/or the platform class loaders do not answer ClassLoader.getDefinedPackage meaningfully.
	 */
	public boolean isPackage(char[][] parentCompoundName, char[] packageName) {
		String pname = getName(parentCompoundName, packageName, '.');
		String sname = pname.replace('.','/');
		boolean r;
		// check in our sources: If we have the folder (this is not 100% according to JLS (7.4ff) but... come on. 
		if (Optional.ofNullable(_findFile(sname)).map(File::isDirectory).orElse(false)) {
			r = true;
		} else
		// check, if our classpath defines it
		if (isPackageOnClasspath(this.cl,pname)) {
			r = true;
		} else
		// Otherwise we use a simple heuristics: If the last segment starts with a lower case character, we assume it is a package
		if (Character.isLowerCase(packageName[0])) {
			r = true;
		} else {
			r = false;
		}
		log(()->(r? "is a ":"not a ")+"package: "+pname);
		// nothing passed: Assume it is not a package 
		return r;
	}

	private boolean isPackageOnClasspath(ClassLoader cl, String pname) {
		if (cl==null || pname==null) {
			return false;
		}
		boolean found = false;
		Boolean r;
		if ((r=this.packages.get(pname))!=null) {
			log(()->"package "+pname+" checked previously");
			found = r;
		} else
		// 
		if (cl.getDefinedPackage(pname)!=null) {
			log(()->"package "+pname+" defined by "+cl);
			found = true;
		} else
		if (cl instanceof IJavaComponentClassLoader) {
			for (ClassLoader l : ((IJavaComponentClassLoader)cl).getParents()) {
				if (isPackageOnClasspath(l, pname)) {
					found = true;
					break;
				}
			}
		} else
		if (isPackageOnClasspath(cl.getParent(), pname)) {
			found = true;
		}
		this.packages.put(pname, r);
		return found;
	}

	private String getName(char[][] parentCompoundName, char[] packageName, char separator) {
		String slashedClassName = (parentCompoundName==null || parentCompoundName.length==0? 
			new String(packageName) :
			new StringBuilder().append(concat(parentCompoundName, separator)).append(separator).append(packageName).toString()
		);
		return slashedClassName;
	}

	// -------

	public static String concat(char[][] compoundName, char separator) {
		if (compoundName==null) {
			return "";
		}
		StringBuilder b = new StringBuilder(200);
		for (char[] s : compoundName) {
			if (b.length() > 0) {
				b.append(separator);
			}
			b.append(s);
		}
		return b.toString();
	}

	private File _findFile(String name) {
		File f = this.sfnToFile.get(name);
		if (f!=null) {
			return f;
		}
		if (this.sourceFolders != null) {
			for (File sf : this.sourceFolders) {
				f = new File(sf, name);
				if (f.exists()) {
					this.sfnToFile.put(name,f);
					return f;
				}
			}
		}
		return null;
	}
	
	private File _findSourceFile(String slashedClassName) {
		return _findFile(slashedClassName+".java");
	}

	private NameEnvironmentAnswer _findType(String slashedClassName) {
		// try as source file first
		File sf = _findSourceFile(slashedClassName);
		if (sf != null) {
			return new NameEnvironmentAnswer(new FileCompilationUnit(sf,this.encoding), null);
		}
		// try as byte array next
		String rn = slashedClassName + ".class";
		InputStream in = this.cl.getResourceAsStream(rn);
		if (in != null) {
			try {
				try {
					return new NameEnvironmentAnswer(ClassFileReader.read(in, rn, true), null);
				} finally {
					in.close();
				}
			} catch (Exception e) {
				if (e instanceof ClassFormatException) {
					int ec = ((ClassFormatException)e).getErrorCode();
					throw new RuntimeException("Failed to read class file (error "+ec+"): " + rn, e);
				}
				throw new RuntimeException("Failed to read class file " + rn, e);
			}
		}
		return null;
	}

	private void log(Supplier<String> s) {
		LOG.finest(s);
	}
}
