/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.java.jdt;
import java.io.File;
import java.io.FileFilter;
import java.lang.management.ManagementFactory;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.jdt.internal.compiler.Compiler;
import org.eclipse.jdt.internal.compiler.DefaultErrorHandlingPolicies;
import org.eclipse.jdt.internal.compiler.env.ICompilationUnit;
import org.eclipse.jdt.internal.compiler.env.INameEnvironment;
import org.eclipse.jdt.internal.compiler.impl.CompilerOptions;
import org.eclipse.jdt.internal.compiler.problem.DefaultProblemFactory;

import com.zfabrik.components.java.LangLevel;


/**
 * A convenience wrapper around the eclipse jdt Java compiler that accepts a source folder, an output and - most importantly - 
 * a class loader as class path (this is the way it should always have been!)
 * @author hb
 */
public class SimpleJDTCompiler {
	private final static Logger logger = Logger.getLogger(SimpleJDTCompiler.class.getName());

	// only and always: Everything is assumed utf-8
	protected final static String UTF8 = "utf-8"; 
	
	private File outRoot;
	private File[] srcs;
	private ClassLoader classPath;
	private boolean successful;

	private static final FileFilter JAVAFILES = new FileFilter() {
		public boolean accept(File pathname) {
			return !pathname.getName().startsWith(".") && (pathname.isDirectory() || pathname.getName().endsWith(".java"));
		}
	};

	public SimpleJDTCompiler(File[] srcs, File outRoot, ClassLoader classPath) {
		super();
		this.srcs = srcs;
		this.outRoot = outRoot;
		this.classPath = classPath;
	}

	public boolean compile() {
		long start = System.currentTimeMillis();
		_delete(this.outRoot);
		this.outRoot.mkdirs();
		Map<String, String> settings = new HashMap<String, String>();

		
		LangLevel ll = LangLevel.determine();
		logger.fine("Using language level "+ll);
		switch (ll) {
		case JAVA11:
			settings.put(CompilerOptions.OPTION_Source,CompilerOptions.VERSION_11);
			settings.put(CompilerOptions.OPTION_TargetPlatform,CompilerOptions.VERSION_11);
			settings.put(CompilerOptions.OPTION_Compliance,CompilerOptions.VERSION_11);
			break;
		case JAVA12:
			settings.put(CompilerOptions.OPTION_Source,CompilerOptions.VERSION_12);
			settings.put(CompilerOptions.OPTION_TargetPlatform,CompilerOptions.VERSION_12);
			settings.put(CompilerOptions.OPTION_Compliance,CompilerOptions.VERSION_12);
			break;
		case JAVA13:
			settings.put(CompilerOptions.OPTION_Source,CompilerOptions.VERSION_13);
			settings.put(CompilerOptions.OPTION_TargetPlatform,CompilerOptions.VERSION_13);
			settings.put(CompilerOptions.OPTION_Compliance,CompilerOptions.VERSION_13);
			break;
		case JAVA14:
			settings.put(CompilerOptions.OPTION_Source,CompilerOptions.VERSION_14);
			settings.put(CompilerOptions.OPTION_TargetPlatform,CompilerOptions.VERSION_14);
			settings.put(CompilerOptions.OPTION_Compliance,CompilerOptions.VERSION_14);
			break;
		case JAVA15:
			settings.put(CompilerOptions.OPTION_Source,CompilerOptions.VERSION_15);
			settings.put(CompilerOptions.OPTION_TargetPlatform,CompilerOptions.VERSION_15);
			settings.put(CompilerOptions.OPTION_Compliance,CompilerOptions.VERSION_15);
			break;
		default:
			throw new IllegalStateException("Unsupported language level: "+ll);
		}
		
		// we enable previews, if the VM has them enabled.
		if (previewsEnabled()) {
			settings.put(CompilerOptions.OPTION_EnablePreviews,CompilerOptions.ENABLED);
		}
		// general compiler options - derived from the VM
		settings.put(CompilerOptions.OPTION_LineNumberAttribute,CompilerOptions.GENERATE);
		settings.put(CompilerOptions.OPTION_SourceFileAttribute,CompilerOptions.GENERATE);
		settings.put(CompilerOptions.OPTION_LocalVariableAttribute,CompilerOptions.GENERATE);
		settings.put(CompilerOptions.OPTION_Encoding,UTF8);
	
		
		CompilerOptions options = new CompilerOptions(settings);
		
		
		INameEnvironment ne = new NameEnvironmentImpl(this.classPath, UTF8, this.srcs);
		CompilerRequestorImpl cr = new CompilerRequestorImpl(this.outRoot);
		ICompilationUnit[] sources = _findSource();
		if (logger.isLoggable(Level.FINE)) {
			logger.fine("Compiling "+sources.length+" source files from "+Arrays.asList(this.srcs)+" to "+this.outRoot);
		}
		org.eclipse.jdt.internal.compiler.Compiler c = new Compiler(
				ne, 
				DefaultErrorHandlingPolicies.exitOnFirstError(), 
				options, 
				cr, 
				new DefaultProblemFactory(Locale.getDefault())
		);
		c.compile(sources);
		long duration = System.currentTimeMillis() - start;
		if (logger.isLoggable(Level.FINE)) {
			logger.fine("Compilation of "+Arrays.asList(this.srcs)+" completed after "+duration+"ms");
		}
		this.successful = !cr.hasError();
		return this.successful;
	}

	private boolean previewsEnabled() {
		List<String> inputArguments = ManagementFactory.getRuntimeMXBean().getInputArguments();
		return inputArguments!=null && inputArguments.contains("--enable-preview");
	}

	public boolean isSuccesful() {
		return this.successful;
	}

	private static void _findFiles(File in, List<File> result, FileFilter filter) {
		if (in.isDirectory()) {
			for (File f : in.listFiles(filter)) {
				_findFiles(f, result, filter);
			}
		} else {
			result.add(in);
		}
	}

	private ICompilationUnit[] _findSource() {
		List<File> files = new LinkedList<File>();
		for (File sf : this.srcs) {
			_findFiles(sf, files, JAVAFILES);
		}
		ICompilationUnit[] result = new ICompilationUnit[files.size()];
		int i = 0;
		for (File f : files) {
			result[i++] = new FileCompilationUnit(f, UTF8);
		}
		return result;
	}

	private static void _delete(File file) {
		if (file.isDirectory()) {
			for (File f : file.listFiles()) {
				_delete(f);
			}
		}
		file.delete();
	}

}
