/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.states;

import static com.zfabrik.impl.components.ComponentDependencyUtil.addDepsFromCSL;
import static com.zfabrik.impl.components.ComponentDependencyUtil.prepareDependencies;

import java.lang.management.ManagementFactory;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.StandardMBean;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.components.IDependencyComponent;
import com.zfabrik.impl.components.ComponentDependencyUtil.IDependencyResolver;
import com.zfabrik.resources.IResourceHandle;
import com.zfabrik.resources.ResourceBusyException;
import com.zfabrik.resources.provider.Resource;
import com.zfabrik.states.ISystemState;
import com.zfabrik.util.expression.X;

/**
 * See {@link ISystemState}.
 * @author hb
 *
 */
public class SystemStateResource extends Resource implements IDependencyComponent, Runnable, ISystemState {
	private final static Logger LOG = Logger.getLogger(SystemStateResource.class.getName());
	private boolean attained;
	private Date lastAttained;
	private final String name;
	private final X findStateParts;
	private ObjectName on;
	
	public SystemStateResource(String name) {
		this.name = name;
		try {
			this.on =  ObjectName.getInstance("zfabrik:type=" + SystemStateResource.class.getName() + ",name=" + name);
		} catch (Exception e) {
			LOG.log(Level.WARNING,"Object Name construction failed. Skipping JMX support.",e);
		}
		this.findStateParts = X.val(this.name).in(X.var(STATES_PARTICIPATION));
	}
	
	public synchronized <T> T as(Class<T> clz) {
		if (IDependencyComponent.class.equals(clz)) {
			return clz.cast(this);
		}
		if (Runnable.class.equals(clz)) {
			return clz.cast(this);
		}
		if (ISystemState.class.equals(clz)) {
			return clz.cast(this);
		} 
		return null;
	}

	@Override
	public void run() {
		this.prepare();
	}
	
	@Override
	public void prepare() {
		_prepareDependencies();
		if (!this.attained) {
			handle().adjust(0, Long.MAX_VALUE, IResourceHandle.STRONG);
			this.attained=true;
			this.lastAttained=new Date();
			this.registerMBean();
			LOG.info("Attained system state: "+this.name);
		}
	}

	@Override
	public void invalidate() throws ResourceBusyException {
		synchronized (this) {
			if (this.attained) {
				this.unregisterMBean();
				LOG.info("Left system state: "+this.name);
			}
			this.attained = false;
		}
	}
	
	private void _prepareDependencies() {
		prepareDependencies(
			handle(),
			new IDependencyResolver() {
				public Collection<String> getDependencies() throws Exception {
					Set<String> deps = new HashSet<String>();
					addDepsFromCSL(deps,handle().as(IComponentDescriptor.class).getProperty(STATES_DEPENDENCY));
					deps.addAll(IComponentsManager.INSTANCE.findComponents(findStateParts));
					return deps;
				}
			},
			LOG
		);
	}

	/**
	 * JMX support
	 */
	public interface SystemStateMBean {
		boolean isAttained();
		Date getLastAttained();
	}
	
	public class SystemStateMBeanImpl implements SystemStateMBean {
		public boolean isAttained() {
			return attained;
		}
		
		public Date getLastAttained() {
			return lastAttained;
		}
	}
	
	/**
	 * Register MBean for the system state
	 */
	private void registerMBean() {
		// register MBEAN
		if (unregisterMBean()) {
			try {
				MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
				mbs.registerMBean(new StandardMBean(new SystemStateMBeanImpl(), SystemStateMBean.class), on);
			} catch (Exception e) {
				LOG.log(Level.WARNING,"MBean registration failed. Skipping JMX support",e);
			}
		}
	}
	
	/**
	 * unregister MBean for the system state
	 */
	private boolean unregisterMBean() {
		if (this.on!=null) {
			try {
				MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
				if (mbs.isRegistered(on)) {
					mbs.unregisterMBean(on);
				}
				return true;
			} catch (Exception e) {
				LOG.log(Level.WARNING,"MBean unregistration failed. Skipping JMX support",e);
			}
		}
		return false;
	}

}
