/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.components.provider.util;

import static com.zfabrik.components.provider.IComponentDescriptorProcessor.EXTENSION_POINT;
import static com.zfabrik.components.provider.IComponentDescriptorProcessor.STYLE_PROCESSED;
import static com.zfabrik.util.expression.X.val;
import static com.zfabrik.util.expression.X.var;

import java.io.IOException;
import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.components.provider.IComponentDescriptorProcessor;
import com.zfabrik.components.provider.props.EvaluationContext;
import com.zfabrik.components.provider.props.Evaluator;
import com.zfabrik.components.provider.props.EvaluationStackOverflowException;
import com.zfabrik.util.runtime.Foundation;

/**
 * Abstract implementation of a component descriptor. Useful in component repository implementations
 *
 * This implementation supports the switch board development mode feature. That is, in development mode,
 * component properties may be overwritten by system properties following the naming scheme:
 * <p><code>
 * &lt; component name &gt;/ &lt; property name &gt; = &lt; new value &gt;
 * </code></p>
 * 
 * This implementation furthermore supports processing of properties for {@link IComponentDescriptorProcessor} implementations, such as
 * the built-in JEXL3 support. 
 */
public abstract class AbstractComponentDescriptor implements IComponentDescriptor, Serializable {

	private final static Logger LOG = Logger.getLogger(AbstractComponentDescriptor.class.getName());
	private static final long serialVersionUID = 1L;
	private Properties rawProperties;
	private transient Properties processedProperties;
	private String name;
	private long revision;

	/**
	 * The identity evaluator simply echos all input.
	 */
	private final static Evaluator IDENTITY_EVALUATOR = new Evaluator() {
		@Override
		public void init(EvaluationContext context) {}
		@Override
		public Object eval(String expression) {
			return expression;
		}
	};
	
	/**
	 * Default constructor
	 */
	public AbstractComponentDescriptor() {}

	/**
	 * Copy constructor
	 */
	public AbstractComponentDescriptor(AbstractComponentDescriptor a) {
		this.setName(a.getName());
		this.setProperties(a.getProperties());
		this.setRevision(a.getRevision());
	}


	//
	// must call this method to set properties
	//
	public synchronized void setProperties(Properties properties) {
		this.rawProperties=new Properties();
		this.rawProperties.putAll(properties);

		// force component name
		this.rawProperties.setProperty(COMPONENT_NAME, this.name);
		
		if (Foundation.MODE_DEVELOPMENT.equals(System.getProperty(Foundation.MODE))) {
			// switch board
			for (Map.Entry<Object,Object> e : properties.entrySet()) {
				String n = e.getKey().toString();
				String v = System.getProperty(this.name+"/"+n);
				if (v!=null) {
					this.rawProperties.setProperty(n, v);
				}
			}
		}
		this.processedProperties = null;
	}

	public Properties getRawProperties() {
		return rawProperties;
	}

	public synchronized void setName(String name) {
		this.name = name;
	}

	public synchronized void setRevision(long revision) {
		this.revision = revision;
	}

	public synchronized String getName() {
		return this.name;
	}

	public synchronized Properties getProperties() {
		if (this.rawProperties == null) {
			this.setProperties(new Properties());
		}
		if (this.processedProperties==null) {
			this.processedProperties = processProperties(this.rawProperties);
		}
		return this.processedProperties;
	}

	/**
	 * This method uses the to process properties to their target representation using none or
	 * some resolvable expression processing facility.
	 */
	public static Properties processProperties(Properties raw) {
		return processProperties(raw,AbstractComponentDescriptor::getProcessorByStyle);
	}
	
	/**
	 * Processing with custom retrieval of processors. See also {@link IComponentDescriptor}.
	 */
	public static Properties processProperties(Properties raw, Function<String, IComponentDescriptorProcessor> getProcessor) {

		if (LOG.isLoggable(Level.FINE)) {
			LOG.fine("Processing properties "+raw);
		}
		
		Map<Object,Object> input = Collections.unmodifiableMap(new HashMap<>(raw));

		// prop expressions
		Map<String,String> nameToExpressions = new HashMap<>();
		// style requested for prop 
		Map<String,String> nameToStyle = new HashMap<>();

		// go and classify and collect them all
		for (Map.Entry<Object, Object> e : input.entrySet()) {
			String name = e.getKey().toString();
			if (e.getValue()!=null) {
				String value = e.getValue().toString();
				int p = name.lastIndexOf(':');
				String style;
				if (p>=0) {
					style = name.substring(p+1);
					name  = name.substring(0,p);
				} else {
					// defaulting
					style = COMPONENT_DESCRIPTOR_STYLE_PLAIN;
				}
				
				// duplicate definition is not supported
				if (nameToStyle.containsKey(name)) {
					throw new IllegalStateException("Duplicate definition of property "+name);
				}
				nameToStyle.put(name,style);
				nameToExpressions.put(name, value);
			}
		}

		// prepare and process
		Map<String,Evaluator> styleToEvaluator = new HashMap<>();
		try {
		
			// do the actual processing by building the context and 
			// ask for any property. Processor may request resolves on their own so that the 
			// actual traversal is intrinsic. Eventually, unless processing was cancelled, all properties will be resolved.
			//
			Map<String,Object> result = new HashMap<>();
			
			boolean[] evaluatorReady = {false};
			
			// it is all driven by the context that is the entry point to all 
			// resolution
			EvaluationContext context = new EvaluationContext() {

				private int stackDepth = 0;
				
				@Override
				public boolean has(String name) {
					// was it seen at all?
					return nameToStyle.containsKey(name);
				}
				
				@Override
				public Object get(String name) {
					if (!evaluatorReady[0]) {
						throw new EvaluationStackOverflowException("Context not initialized yet. Note: You may not attempts resolves during Evaluator.init(...)");
					}
					stackDepth++;
					if (stackDepth>EvaluationStackOverflowException.MAX_STACK_DEPTH) {
						throw new EvaluationStackOverflowException("Exceeded max evaluation depth of "+EvaluationStackOverflowException.MAX_STACK_DEPTH);
					}
					try {
						// This is the crucial thing: If we don't have it yet, go and ask the right evaluator
						// 
						if (result.containsKey(name)) {
							// already resolved
							return result.get(name);
						}
						
						// do the thing!
						String style = nameToStyle.get(name);
						if (style==null) {
							// what we are asked for is not in the props
							return null;
						}
						String expression = nameToExpressions.get(name);
						try {
							Object val = styleToEvaluator.get(style).eval(expression);
							if (LOG.isLoggable(Level.FINER)) {
								if (!COMPONENT_DESCRIPTOR_STYLE_PLAIN.equals(style)) {
									LOG.finer("Resolved "+name+":"+nameToStyle.get(name)+"="+expression+" to value \""+Objects.toString(val)+"\"");
								}
							}
							result.put(name, val);
							return val;
						} catch (Exception e) {
							throw new IllegalStateException("Evaluation error for ("+name+":"+nameToStyle.get(name)+"="+expression+"): "+e.getMessage(),e);
						}
					} finally {
						stackDepth--;
					}
				}
				
				@Override
				public Map<Object, Object> getRawProperties() {
					return input;
				}
				
			};

			// now get and initialize the evaluators
			for (String style : nameToStyle.values()) {
				Evaluator evaluator;
				if (!styleToEvaluator.containsKey(style)) {
					if (COMPONENT_DESCRIPTOR_STYLE_PLAIN.equals(style)) {
						// hard-coded case: Identity evaluator
						evaluator = IDENTITY_EVALUATOR;
					} else {
						// regular case
						IComponentDescriptorProcessor processor = getProcessor.apply(style);
						if (processor==null) {
							throw new IllegalStateException("No processor provided for style \""+style+"\"");
						}
						evaluator = processor.createEvaluator();
						if (evaluator==null) {
							throw new IllegalStateException("No evaluator provided for style \""+style+"\"");
						}
					}
					evaluator.init(context);
					styleToEvaluator.put(style, evaluator);
				}
			}
			
			// ready to roll..
			evaluatorReady[0]=true;

			// resolve it all by asking the context for resolution
			Properties properties = new Properties();
			for (Map.Entry<String,String> e : nameToStyle.entrySet()) {
				String name = e.getKey();
				String style = e.getValue();
				Object v = context.get(name);
				if (v==null) {
					if (LOG.isLoggable(Level.FINE)) {
						if (!COMPONENT_DESCRIPTOR_STYLE_PLAIN.equals(style)) {
							LOG.fine("Property expression "+name+":"+style+"="+raw.getProperty(name)+" resolved to null and will be ignored");
						}
					}
				} else {
					properties.put(name,v);
				}
			}
			
			return properties;
		} catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException("Failed to process properties",e);
		} finally {
			// close all evaluators
			for (Map.Entry<String,Evaluator> e : styleToEvaluator.entrySet()) {
				try {
					e.getValue().close();
				} catch (Exception ex) {
					LOG.log(Level.WARNING, "Problem closing evaluator for \""+e.getKey()+"\"", ex);
				}
			}
		}
	}

	/**
	 * Return a processor by type. Never returns null
	 */
	private static IComponentDescriptorProcessor getProcessorByStyle(String style) {
		try {
			Collection<String> processors = IComponentsManager.INSTANCE.findComponents(
				val(EXTENSION_POINT).in(var(EXTENSION_POINTS))
				.and(
				  val(style).in(var(STYLE_PROCESSED))
				)
			);
			if (processors.isEmpty()) {
				throw new IllegalStateException("No "+IComponentDescriptorProcessor.class.getName()+" found for style "+style);
			}
			if (processors.size()>1) {
				throw new IllegalStateException("Found "+processors.size()+" "+IComponentDescriptorProcessor.class.getName()+" implementations for style "+style+" (only one is supported)");
			}
			String processorName = processors.iterator().next();
			IComponentDescriptorProcessor processor = IComponentsLookup.INSTANCE.lookup(processorName, IComponentDescriptorProcessor.class);
			if (processor==null) {
				throw new IllegalStateException("Lookup for "+processorName+" returned no instance");
			}
			return processor;
		} catch (IOException e) {
			 throw new RuntimeException("Failed to find "+IComponentDescriptorProcessor.class.getName()+" for style "+style,e);
		}
	}

	public synchronized long getRevision() {
		return this.revision;
	}

	public synchronized String getType() {
		if (this.rawProperties!=null) {
			return this.getProperties().getProperty(IComponentDescriptor.COMPONENT_TYPE);
		}
		return null;
	}


	@Override
	public synchronized String getProperty(String name) {
		return this.getProperties().getProperty(name);
	}

	@Override
	public int hashCode() {
		return (this.rawProperties!=null? this.rawProperties.hashCode():0) ^
			   (this.name!=null? this.name.hashCode() : 0) ^
			   (Long.valueOf(this.revision).hashCode());
	}

	@Override
	public boolean equals(Object obj) {
		if (this==obj) {
			return true;
		}
		if (obj instanceof AbstractComponentDescriptor) {
			AbstractComponentDescriptor d = (AbstractComponentDescriptor) obj;
			return 	eq(this.rawProperties,d.rawProperties) &&
					eq(this.name,d.name) &&
					this.revision==d.revision;
		}
		return false;
	}

	private boolean eq(Object o1, Object o2) {
		if (o1==null) {
			return o2==null;
		}
		return o1.equals(o2);
	}
}
