/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.components.provider.util;

import java.io.File;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.management.ObjectName;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.components.provider.IComponentsRepository;
import com.zfabrik.components.provider.IComponentsRepositoryContext;
import com.zfabrik.resources.provider.Resource;
import com.zfabrik.util.expression.X;
import com.zfabrik.util.fs.FileUtils;
import com.zfabrik.util.runtime.Foundation;
import com.zfabrik.util.sync.ISynchronization;
import com.zfabrik.util.sync.ISynchronizer;

/**
 * Abstract implementation of a component repository. This is by far the
 * easiest way of implementing a component repository. The only methods needed
 * to implement are
 * <ol>
 * <li>{@link #scan(DB)} to compute an update of the available components, their properties and revisions</li>
 * <li>{@link #download(FSCRDC, File)} to download a component's resources, if any, to a local file system folder</li>
 * </ol>
 * where FSCRC extends FSCRDBComponent and DB extends FSComponentRepositoryDB&lt;FSCRC&gt;.
 *
 *
 * @author hb
 *
 */
public abstract class AbstractExtComponentRepository<FSCRC extends FSCRDBComponent, DB extends FSComponentExtRepositoryDB<FSCRC>> implements IComponentsRepository, ISynchronizer {
	private static final String REVISION = "r";

	public static Map<String,AbstractExtComponentRepository<?,?>> ALL = Collections.synchronizedMap(new HashMap<String, AbstractExtComponentRepository<?,?>>());

	private String name;
	private int prio;
	private FSComponentRepositoryHelper<DB> helper;
	private Class<DB> clz;
	private DB db,newDb;
	private IComponentsRepositoryContext context;
	private boolean inited;
	private int syncs;
	private ObjectName on;
	private String repoName;
	private File root;

	public AbstractExtComponentRepository(
			String name,
			Class<DB> clz
	) {
		this.clz = clz;
		this.name = name;
	}

	/**
	 * Finish configuration. Must be called before using as repo and using root (not roots!) information. I.e. this should
	 * be called as early as possible BUT after determining the expected configuration that is 
	 * used to decide whether a repo cache is to be evicted.
	 *
	 * @param prio Repository priority. See {@link IComponentsRepository}.
	 * @param evictionDelay Delay of eviction for old component versions. This should be so high that concurrent home usage may not really happen. Defaults to 24h.
	 * @param repoKey A key of the repo that identifies its cache space. For different usages, same name, repo key should. For example the dev repo may be used for different workspaces and is still one repo
	 */
	protected void configure(int prio, Long evictionDelay, String repoKey) {
		this.prio = prio;
		this.repoName = repoKey;
		this.root = FileUtils.computeSafePath(new File(System.getProperty(Foundation.HOME_LAYOUT_REPOS)),this.repoName.replace('/', '_'));
		this.helper = new FSComponentRepositoryHelper<DB>(
			clz,
			this.root,
			getExpectedConfiguration()
		);
		if (evictionDelay!=null) {
			this.helper.setEvictionDelay(evictionDelay);
		}
	}
	
	/**
	 * provided expected repository configuration. This property set is used by the 
	 * {@link FSComponentRepositoryHelper} to make a decision on whether the repository
	 * cache is still valid. Any change in repository configuration between repository
	 * initialization will be taken as reason to completely purge the local repository. 
	 * By default this method returns the complete repository configuration.
	 */
	protected Properties getExpectedConfiguration() {
		IComponentDescriptor d = IComponentsManager.INSTANCE.getComponent(name);
		if (d==null) {
			throw new IllegalStateException("Component repository "+this.name+" not found");
		}
		return d.getProperties();
	}

	/**
	 * Is actually <code>configure(prio,evictionDelay,this.name)</code>.
	 * @param prio Repository priority. See {@link IComponentsRepository}.
	 * @param evictionDelay Delay of eviction for old component versions. This should be so high that concurrent home usage may not really happen. Defaults to 24h.
	 */
	protected void configure(int prio, Long evictionDelay) {
		configure(prio,evictionDelay,this.name);
	}

	/**
	 * Is actually <code>configure(prio,null)</code>.
	 * @param prio Repository priority. See {@link IComponentsRepository}.
	 */
	protected void configure(int prio) {
		configure(prio, null);
	}


	public String getName() {
		return name;
	}

	protected File getCacheRoot() {
		return root;
	}

	/**
	 * Set eviction of old component copies
	 */
	protected void setEvictionDelay(long evictionDelay) {
		if (this.helper==null) {
			throw new IllegalStateException("Need to configure first");
		}
		this.helper.setEvictionDelay(evictionDelay);
	}


	/**
	 * Simple type helper for {@link Resource} implementations. Call this to check whether
	 * delegation to the component repository should be implemented
	 */
	public static boolean has(Class<?> clz) {
		return IComponentsRepository.class.equals(clz) || ISynchronizer.class.equals(clz);
	}

	/**
	 * Simple type helper for {@link Resource} implementations. Call this to check whether
	 * delegation to the component repository should be implemented
	 */
	public <T> T as(Class<T> clz) {
		return clz.cast(this);
	}

	/**
	 * gets the repository context
	 */
	public IComponentsRepositoryContext getContext() {
		return context;
	}

	/**
	 *  Stop and unregister the repo. The instance becomes unusable after that
	 */
	public synchronized void stop() {
		if (inited) {
			try {
				logger.fine("unregistering "+this);
				IComponentsManager.INSTANCE.unregisterRepository(this);
			} finally {
				try {
					if (this.on!=null) {
						ManagementFactory.getPlatformMBeanServer().unregisterMBean(this.on);
						this.on=null;
					}
				} catch (Exception e) {
					logger.log(Level.WARNING, "Mbean unregistration error", e);
				} finally {
					this.inited=false;
					ALL.remove(this.name);
				}
			}
		}
	}

	/**
	 * Start the repository and register it.
	 */
	public synchronized void start() {
		if (!inited) {
			//
			// #896 First prepare some state. If we register before that,
			// checkDB will lead us into recursion which will
			// give us Overlappingfilelock exceptions. So in this case,
			// we do always have a DB when someone asks us.
			//
			try {
				this.checkDB();
			} catch (IOException ioe) {
				throw new RuntimeException("Failed to init repository: "+name,ioe);
			}
			//
			// register and complete
			//
			this.context = IComponentsManager.INSTANCE.registerRepository(this.prio,this);
			ALL.put(this.name,this);
			try {
				this.on = ObjectName.getInstance("zfabrik:type="+AbstractExtComponentRepository.class.getName()+",name=" + this.name);
				ManagementFactory.getPlatformMBeanServer().registerMBean(this.repoMbean,this.on);
			} catch (Exception e) {
				throw new IllegalStateException("Mbean registration for repo failed: "+this.name, e);
			}
			this.inited=true;
			logger.fine("Registered "+this);
		}
	}

	/**
	 * pre invalidation collection of outdated components
	 */
	public synchronized void preInvalidation(ISynchronization sync) {
		try {
			syncs++;
			Lock lock = this.helper.lockDB();
			try {
				// delta map
				Map<String,FSCRDBComponent> x = new HashMap<String, FSCRDBComponent>();
				if (this.db!=null) {
					x.putAll(this.db.getComponents());
				}
				logger.fine("preInvalidation Scan on "+this.name);
				DB somedb = scan(this.db);

				// check for differences
				for (Map.Entry<String,? extends FSCRDBComponent> e : somedb.getComponents().entrySet()) {
					String cn = e.getKey();
					FSCRDBComponent now = x.remove(cn);
					if (now==null || !now.equals(e.getValue())) {
						// new component (now == null) or updated revision (unequals)
						sync.getInvalidationSet().add(IComponentsLookup.COMPONENTS+"/"+cn);
					}
				}
				// anything left in current db is gone from the repo. Invalidate
				if (!x.isEmpty()) {
					for (Map.Entry<String,FSCRDBComponent> e : x.entrySet()) {
						sync.getInvalidationSet().add(IComponentsLookup.COMPONENTS+"/"+e.getKey());
					}
				}

				// any module that is not managed anymore is to be invalidated as a whole
				// that is, we add all components from modules that moved out of
				// scope to the invalidation set.
				Set<String> modules = new HashSet<String>(this.db.getModules());
				modules.removeAll(somedb.getModules());
				for (String m : modules) {
					for (String cn : this.db.getComponentsOfModule(m)) {
						sync.getInvalidationSet().add(IComponentsLookup.COMPONENTS+"/"+cn);
					}
				}

				// give derivations a chance
				preInvalidation(this.db,somedb,sync);

				// put new db for later assignment
				// in complete we reset the DB and on home
				// we simply re-use the one computed here to save
				// another scan (#909)
				this.setNewDb(somedb);
				if (!somedb.equals(this.db)) {
					// save if modified
					logger.fine("saveDB: "+this.name);
					this.helper.saveDB(somedb);
				}
			} finally {
				lock.close();
			}
		} catch (Exception e) {
			throw new RuntimeException("Failure in preInvalidation step: "+this.name,e);
		}
	}

	/**
	 * Can be overridden to add additional invalidation behavior based on the current DB and
	 * the new DB. Standard behavior has already applied when this method is called
	 * @param currentDB
	 * @param newDB
	 * @param sync
	 */
	protected void preInvalidation(DB currentDB, DB newDB, ISynchronization sync) {
	}

	@Override
	public synchronized void complete(ISynchronization sync) {
		if (Foundation.isWorker()) {
			// force reload in check db
			this.db = null;
		} else {
			// #967: If we are on home and newdb was not computed, do NOT reset as that
			// would force a rescan upon verification
			if (newDb!=null) {
				//
				// reset the db, so that we re-assign in check db
				//
				db = null;
			}
		}
	}

	//
	// IComponentRepository
	//

	@Override
	public synchronized long getRevision(String cn, boolean localOnly) throws IOException {
		this.checkDB();
		FSCRDBComponent c = this.db.getComponents().get(cn);
		if (c!=null)
			return c.getRevision();
		else if (! localOnly && this.db.delegate(cn)) {
			IComponentsRepository r = this.context.next();
			if (r!=null) {
				return r.getRevision(cn);
			}
		}
		return -1;
	}

	@Override
	public synchronized Collection<String> findComponents(X x, boolean localOnly) throws IOException {
		Collection<String> cs;

		this.checkDB();
		IComponentsRepository r = this.context.next();
		if (! localOnly && r != null) {
			cs = r.findComponents(x);
			Iterator<String> it = cs.iterator();
			while (it.hasNext()) {
				if (!this.db.delegate(it.next())) {
					it.remove();
				}
			}
			cs.removeAll(this.db.getComponents().keySet());

		} else {
			cs = new HashSet<String>();

		}
		cs.addAll(this.db.findComponents(x));
		return cs;
	}

	@Override
	public synchronized IComponentDescriptor getComponent(String cn, boolean localOnly) {
		try {
			this.checkDB();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
		IComponentDescriptor res = this.db.getComponents().get(cn);
		if (res != null || localOnly) {
			return res;

		} else if (this.db.delegate(cn)) {

			IComponentsRepository r = this.context.next();
			if (r !=null) {
				res = r.getComponent(cn);
			}
			return res;
		}
		return null;
	}

	@Override
	public synchronized File retrieve(String cn, boolean localOnly) throws IOException {
		this.checkDB();
		FSCRC c = this.db.getComponents().get(cn);
		if (c==null && !localOnly) {
			// we don't have it and we are supposed to look deeper
			if (this.db.delegate(cn)) {
				IComponentsRepository next = this.context.next();
				if (next!=null) {
					return next.retrieve(cn);
				}
			}
		}
		if (c==null) {
			// local only and don't have it
			return null;
		}

		File folder = this.helper.getComponentFolder(c);
		// check if resolved already (i.e. we provided it before)
		if (c.isResolved()) {
			return folder;
		}

		// actually do retrieve
		if (c.isHasFolder()) {
			Lock clock = this.helper.lockComponent(c.getName());
			try {
				String rs = Long.toString(c.getRevision());
				if (!rs.equals(clock.properties().get(REVISION))) {
					// still need to downloaded
					// remove previous
					this.helper.evictLocalComponent(c.getName());
					// download
					logger.fine("Downloading component "+c.getName()+": "+this.name);
					long start = System.currentTimeMillis();
					download(c, folder);
					long duration = System.currentTimeMillis()-start;
					logger.fine("Download of component "+c.getName()+" completed after "+duration+"ms: "+this.name);
					clock.properties().put(REVISION,rs);
					clock.update();
				}
			} finally {
				clock.close();
			}
		}
		// No error. Mark as resolved and return folder
		c.setResolved(true);
		return folder;
	}

	@Override
	public synchronized Set<String> getModules(boolean localOnly) throws IOException {
		this.checkDB();
		if (this.context.next()==null || localOnly) {
			return this.db.getModules();
		} else {
			HashSet<String> s = new HashSet<String>(this.context.next().getModules(false));
			s.addAll(this.db.getModules());
			return s;
		}
	}

	//
	// to be implemented for monitoring
	//
	/**
	 * Overall revision of the repository - if available
	 */
	protected long getRevision() {
		return -1;
	}

	/**
	 * Some URL style information on the external data source the repository implementation relies on
	 */
	protected String getURL() {
		return null;
	}

	/**
	 * Operational mode checks
	 */
	protected boolean isRelaxedMode() {
		return IComponentsRepository.COMPONENT_REPO_MODE_RELAXED.equalsIgnoreCase(System.getProperty(IComponentsRepository.COMPONENT_REPO_MODE));
	}

	/**
	 * Check offline mode and if so throw an exception. This is to used in repository implementations in conjunction with the relaxed
	 * mode. That is: We consider the offline mode as one particular failure mode when trying to access remote resources. Only when in relaxed mode and
	 * sensible data is present, continuing in offline mode is advisable - as for any other remote access failure.
	 */
	protected void checkOfflineMode() {
		if (Foundation.isOfflineMode()) {
			throw new OfflineModeException();
		}
	}



	//
	// Methods to actually implement
	//

	/**
	 * Provide an updated DB. This DB will be used to compute
	 * invalidations on synchronization and it will be used
	 * to serve component meta-data.
	 * If the repository implementation needs to attach additional
	 * meta-data to the DB, it should overwrite the DB class to do
	 * so, as the result of a scan potentially will be discarded.
	 *
	 */
	public abstract DB scan(DB current);

	/**
	 * Download resources of a single component into the given folder
	 */
	public abstract void download(FSCRC component, File folder);



	//
	// util
	//

	/**
	 * returns the current DB
	 */
	public final synchronized DB getDB() throws IOException {
		this.checkDB();
		return this.db;
	}

	//
	private synchronized void checkDB() throws IOException {
		if (this.helper==null) {
			throw new IllegalStateException("Call configure(..) before use of "+AbstractExtComponentRepository.class.getName());
		}
		if (this.db==null) {
			Lock l = this.helper.lockDB();
			try {
				if (!Foundation.isWorker()) {
					if (this.newDb!=null) {
						// a new db has been computed during sync
						logger.fine("assign new db: "+this.name);
						this.db = this.newDb;
						this.newDb = null;
					} else {
						// try to read one
						DB pdb = null;
						try {
							logger.fine("readDB: "+this.name);
							pdb = this.helper.readDB();
						} catch (Exception e) {
							logger.log(Level.WARNING,"Failed to read db, will compute new one: "+this.name,e);
						}
						// update or compute a new one in case of error
						logger.fine("checkDB Scan on "+this.name);
						this.db = scan(pdb);
						logger.fine("saveDB: "+this.name);
						this.helper.saveDB(this.db);
					}
					//
					// #888 check for module overrides
					//
					IComponentsRepository next = (this.context!=null? this.context.next() : null);
					if (next!=null) {
						Set<String> mods = new HashSet<String>(this.getModules(true));
						mods.retainAll(next.getModules(false));
						for (String m : mods) {
							logger.info("Overriding module "+m+": "+this.name);
						}
					}

				} else {
					try {
						logger.fine("readDB: "+this.name);
						this.db = this.helper.readDB();
						if (this.db==null) {
							throw new IllegalStateException("Repository DB not found: "+name);
						}
					} catch (IOException ioe) {
						throw new RuntimeException("Failed to load repository DB: "+name, ioe);
					}
				}
			} finally {
				l.close();
			}
		}
	}


	// park a new db for later assignment
	private synchronized void setNewDb(DB db) {
		logger.fine("set new db: "+this.name);
		this.newDb = db;
	}

	@Override
	public String toString() {
		return "component:"+this.name+",repo:"+this.repoName+",prio:"+this.prio+",mode:"+(isRelaxedMode()?IComponentsRepository.COMPONENT_REPO_MODE_RELAXED:IComponentsRepository.COMPONENT_REPO_MODE_STRICT);
	}

	// ------- jmx --------------

	public static interface RepoMBean {
		//
		int getQueryCacheSize();
		int getQueryCacheQueries();
		int getQueryCacheHits();
		//
		long getRepoRevision();
		int  getNumberSynchronizations();
		int  getNumberComponents();
		long getImplementationRevision();
	}

	public class Repo implements RepoMBean {
		public long getImplementationRevision() {
			return FSComponentExtRepositoryDB.serialVersionUID;
		}
		public int getNumberComponents() {
			synchronized (AbstractExtComponentRepository.this) {
				if (db!=null) {
					return db.getComponents().size();
				}
			}
			return -1;
		}
		public int getNumberSynchronizations() {
			synchronized (AbstractExtComponentRepository.this) {
				return syncs;
			}
		}
		public int getQueryCacheHits() {
			synchronized (AbstractExtComponentRepository.this) {
				if (db!=null) {
					return db.getCacheHits();
				}
			}
			return -1;
		}
		public int getQueryCacheQueries() {
			synchronized (AbstractExtComponentRepository.this) {
				if (db!=null) {
					return db.getCacheQueries();
				}
			}
			return -1;
		}
		public int getQueryCacheSize() {
			synchronized (AbstractExtComponentRepository.this) {
				if (db!=null) {
					return db.getCacheSize();
				}
			}
			return -1;
		}
		public long getRepoRevision() {
			synchronized (AbstractExtComponentRepository.this) {
				return AbstractExtComponentRepository.this.getRevision();
			}
		}
		public String getURL() {
			synchronized (AbstractExtComponentRepository.this) {
				return AbstractExtComponentRepository.this.getURL();
			}
		}
	}
	private Repo repoMbean = new Repo();

	private final static Logger logger = Logger.getLogger(AbstractExtComponentRepository.class.getName());

	// test support
	public void test_setInited(boolean inited) {
		this.inited = inited;
	}
	public void test_setContext(IComponentsRepositoryContext context) {
		this.context = context;
	}
	public void test_setDb(DB db) {
		this.db = db;
	}
}
