/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.components.provider.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.RandomAccessFile;
import java.io.Writer;
import java.nio.channels.FileLock;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.components.provider.util.Lock;

/**
 * A file system lock with properties. This is used by component repository implementations to manage concurrency 
 * in update decisions of locally managed resources.
 * The typical usage of this class is 
 * <pre>
 * LockingRevFile lrf = new LockingRevFile(f);
 * lrf.open();
 * try {
 *    // do some checking based on lrf.properties()
 *    // do whatever updates needed
 *    // update lrf.properties() as required
 *    // persist updates
 *    lrf.update();
 * } finally {
 *   lrf.close();
 * }
 * </code>
 * 
 * @author hb
 *
 */
public class LockingRevFile implements Lock {
	private File file;
	private FileLock lock;
	private RandomAccessFile raf;
	private Properties props;
//	private static DateFormat fmt = DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM);
	
	public LockingRevFile(File file) {
		this.file = file;
	}
	
	public boolean exists() {
		return this.file.exists();
	}
	
	public synchronized void open() throws IOException {
		this.file.createNewFile();
		this.raf = new RandomAccessFile(this.file,"rw");
		boolean suc = false;
		try {
			this.lock = this.raf.getChannel().lock();
			this.props = new Properties();
			byte[] r = new byte[(int) this.raf.length()];
			this.raf.read(r);
			this.props.load(new ByteArrayInputStream(r));
			suc = true;
			if (logger.isLoggable(Level.FINE)) {
				logger.fine("Locked on file "+this.file.getAbsolutePath());
			}
		} finally {
			if (!suc) {
				try {
					this.raf.close();
				} finally {
					this.raf = null;
				}
			}
		}
	}
	
	public synchronized Properties properties() {
		if (this.raf==null)
			throw new IllegalStateException("not opened yet");
		return this.props;
	}
	
	public synchronized void update() throws IOException {
		if (this.raf==null)
			throw new IllegalStateException("not opened yet");
		this.raf.seek(0);
		ByteArrayOutputStream baOut = new ByteArrayOutputStream();
		Writer w = new OutputStreamWriter(baOut);
		this.props.store(w,null);
		w.close();
		byte[] r = baOut.toByteArray();
		this.raf.write(r);
		this.raf.setLength(r.length);
		if (logger.isLoggable(Level.FINE)) {
			logger.fine("Updated lock file "+this.file.getAbsolutePath());
		}
	}
	public synchronized void close() throws IOException {
		if (this.raf==null)
			return;
		try {
			this.lock.release();
			if (logger.isLoggable(Level.FINE)) {
				logger.fine("Released lock on "+this.file.getAbsolutePath());
			}
		} finally {
			try {
				this.raf.close();
			} finally {
				this.raf = null;
				this.lock = null;
			}
		}
	}
	

	private final static Logger logger = Logger.getLogger(LockingRevFile.class.getName());
}
