/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.launch;

import java.io.File;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.concurrent.Callable;


/**
 * This utility class is used to initialize a z2 core within another Java process. In other words this 
 * class allows to use z2 embedded. 
 * <p>
 * This class can be used on its own and is provided with the jar library <b>z.jar</b> that can be found in the 
 * <code>bin</code> folder of a z2 home installation.
 * <p>
 * There are some important considerations and pre-requisites to run z2 embedded successfully:
 * <ol>
 * <li>The location of a z2 home installation must be set be either setting the system property <code>com.zfabrik.home</code> or by specifying an environment property <code>Z2_HOME</code>
 * pointing to a z2 home installation. This is required so that the root repository information (in <code>Z2_HOME/bin/runtime.properties</code>) local repository
 * caches (in <code>Z2_HOME/work</code>) can be found. The system property takes precedence over the environment variable.
 * </li>
 * <li>Many control flows in z2 require a WorkUnit on the current thread. In order to execute code safely within the z2 environment, encapsulate it as a Runnable or a 
 * Callable and use the work methods below. 
 * </li>
 * </ol> 
 * 
 * In order to lookup z2 hosted components without having the core implementation on the class path, the Resource Management JNDI provider may be used. A typical
 * invocation of an embedded z2 environment looks like this: 
 * <pre>
 * public static void main(String[] args) throws Exception {
 * 	ProcessRunner.start();
 * 	ProcessRunner.work(new Callable<Void>() {
 * 		public Void call() throws Exception {
 *			((Runnable) new InitialContext().lookup("components:&lt;my component&gt;?type="+Runnable.class.getName())).run();
 *			return null;
 *		}
 *	});
 *	ProcessRunner.stop();
 * }
 * </pre>
 * 
 * Process running is cumulative. Repeated start/stop calls will be counted and only the outer calls are really effectiv.
 * 
 * 
 * @author hb
 *
 */
public class ProcessRunner {
	public static final String Z2_HOME = "Z2_HOME";
	public static final String COM_ZFABRIK_HOME = "com.zfabrik.home";

	private static Method workRunnable, workCallable, start, stop, beginWork, closeWork, setRollbackOnly;
	private static int count=0;
	
	private static boolean inited = false;

	private ProcessRunner() {}
	
	/**
	 * Initialize the z2 environment in this process. Repeated calls will be ignored.
	 */
	public static synchronized void start() {
		try {
			if (count==0) { 
				if (!inited) {
					String sZ2Home = System.getProperty(COM_ZFABRIK_HOME);
					if (sZ2Home==null) {
						sZ2Home = System.getenv(Z2_HOME);
						if (sZ2Home==null) {
							throw new IllegalArgumentException("Missing property "+COM_ZFABRIK_HOME+" or environment variable "+Z2_HOME);
						}
					}
					File zjar = new File(new File(sZ2Home),"bin/z.jar");
					ClassLoader cl = new URLClassLoader(new URL[]{zjar.toURI().toURL()});
					// get the internal process runner
					Class<?> clz = Class.forName("com.zfabrik.launch.impl.ProcessRunnerImpl", false, cl);
					// start!
					start = clz.getMethod("start");
					workCallable = clz.getMethod("work", Callable.class);
					workRunnable = clz.getMethod("work", Runnable.class);
					beginWork = clz.getMethod("beginWork");
					closeWork = clz.getMethod("closeWork");
					setRollbackOnly = clz.getMethod("setRollbackOnly");
					stop = clz.getMethod("stop");
					inited = true;
				}
				start.invoke(null);
				count=1;
			} else {
				count++;
			}
		} catch (Exception e) {
			throw new RuntimeException("Failed to start z2 process", e);
		}
	}

	/**
	 * Tasks in z2 generally require an association with a work unit (an abstract
	 * transaction).
	 * 
	 * In order to execute code safely within the z2 environment, encapsulate it as a {@link Runnable} or a {@link Callable} and
	 * use the {@link #work} methods.    
	 */
	public static void work(Runnable r) {
		try {
			_check();
			workRunnable.invoke(null, r);
		} catch (Exception e) {
			throw new RuntimeException("work failed",e);
		}
	}

	/**
	 * Tasks in z2 generally require an association with a work unit (an abstract
	 * transaction).
	 * 
	 * In order to execute code safely within the z2 environment, encapsulate it as a {@link Runnable} or a {@link Callable} and
	 * use the {@link #work} methods.    
	 */
	@SuppressWarnings("unchecked")
	public static <T> T work(Callable<T> c) {
		_check();
		try {
			return (T) workCallable.invoke(null,c);
		} catch (Exception e) {
			throw new RuntimeException("work failed",e);
		}
	}

	/**
	 * Begin a unit of work. Make your to close the unit of work again. 
	 * Better use the work methods, unless you are really required to 
	 * spread begin and end of a unit of work to different points in the control flow.
	 */
	public static void beginWork(){
		_check();
		try {
			beginWork.invoke(null);
		} catch (Exception e) {
			throw new RuntimeException("begin work failed",e);
		}
	}

	/**
	 * End a unit of work.  
	 * Better use the work methods, unless you are really required to 
	 * spread begin and end of a unit of work to different points in the control flow.
	 */
	public static void closeWork(){
		_check();
		try {
			closeWork.invoke(null);
		} catch (Exception e) {
			throw new RuntimeException("close work failed",e);
		}
	}

	/**
	 * Flag a unit of work as rollback only. I.e. the work will not
	 * be committed at the close of the unit of work but rather rolled
	 * back (if there is any).
	 */
	public static void setRollBackOnly(){
		_check();
		try {
			setRollbackOnly.invoke(null);
		} catch (Exception e) {
			throw new RuntimeException("setRollbackOnly failed",e);
		}
	}
	
	/**
	 * check if the process is already initialized 
	 */
	public synchronized boolean isRunning() {
		return count>0;
	}
	
	
	private static void _check() {
		if (count==0) {
			throw new IllegalStateException("The z2 environment has not been initialized. Call ProcessRunner.start() first.");
		}
	}

	/**
	 * stop the z2 environment in this process. 
	 */
	public static synchronized void stop() {
		if (count==1) {
			try {
				// stop
				stop.invoke(null);
			} catch (Exception e) {
				throw new RuntimeException("Error during shutdown of z2 process", e);
			} finally {
				count=0;
			}
		} else {
			count--;
		}
	}

}
