/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.resources.util;

import java.lang.ref.Reference;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.lang.ref.WeakReference;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.TreeSet;

/**
 * A map wrapper that manages a map over expirable values.
 *
 * Note: if an invalidation queue has been set, this map will NOT remove invalidated values from the
 * map but relies on external processing to do so.
 * 
 * 
 *
 * @author hb
 *
 * @param &lt;K&gt; key
 * @param &lt;V&gt; Value to hold (possibly by weak ref)
 * @param &lt;U&gt; Extra data associated with V but not determining its life cycle
 */
public class ExpirableValuesPseudoMap<K, V, U> {
	// management type per value
	public final static short HARD = 2;
	public final static short SOFT = 1;
	public final static short WEAK = 0;

	// Garbage Collection queue, to find those for which the resource has been collected
	private ReferenceQueue<V> gcQueue = new ReferenceQueue<V>();
	
	// Some values in the map may be subject to 
	// Timing considerations, such as expiration and ttl. 
	// For those we maintain a nextEvent ordered view by a TreeSet.
	private Set<ValueHolder> timedValueHolders = new TreeSet<>((vh1,vh2)->{
		// this comparator is super important, as it also defines equality
		// that is why, before changing the result of nextEvent, we must remove
		// the entry from the set
		int res = Long.compare(vh1.nextEvent(), vh2.nextEvent());
		if (res==0) {
			// we want to say it is equal, only when it is really equal
			if (vh1 == vh2) {
				return 0;
			} else {
				// some order, doesn't matter
				return 1;
			}
		}
		return res;
	});

	// stats
	private int maxSize;
	private int intMaxSize;
	private int shrinks;

	public void resetStats() {
		this.shrinks = 0;
		this.maxSize = this.backingmap.size();
		this.intMaxSize = this.backingmap.size();
	}

	public int getMaxSize() {
		return maxSize;
	}

	public int getShrinks() {
		return shrinks;
	}
	
	// get suggested next clean out time
	public long getNextTime() {
		return nextTime;
	}



	// 
	// value holder (is the real in-memory key for managed resources
	//
	public class ValueHolder {

		// WeakValueHolder
		private class WeakValueHolderRef extends WeakReference<V> {
			public WeakValueHolderRef(V value) {
				super(value,gcQueue);
			}
			public ValueHolder getValueHolder() {
				return ValueHolder.this;
			}
		}

		// SoftValueHolder
		private class SoftValueHolderRef extends SoftReference<V> {
			public SoftValueHolderRef(V value) {
				super(value,gcQueue);
			}
			public ValueHolder getValueHolder() {
				return ValueHolder.this;
			}
		}

		// time to live. If set >0 the a hard ref will be kept, as long for
		// ttl millis whenever the value holder is touched (implicit in adjust)
		private long ttl =0;
		// when do we need to release the ttl implied hard ref
		private long untouch = Long.MAX_VALUE;
		// when do we expire
		private long expiration = Long.MAX_VALUE;
		private boolean timed;
		private Reference<V> ref;
		private V value;
		private K key;
		private U extra;

		public ValueHolder(K key, V value) {
			this(key, value,0,Long.MAX_VALUE,WEAK);
		}

		public ValueHolder(K key, V value,long ttl,long exp, short rt) {
			this.value = value;
			this.key = key;
			this.adjust(ttl, exp, rt);
		}

		public ValueHolder adjust(long ttl,long exp, short rt) {
			// remove if timely before
			this.dequeue();
			this.ttl = ttl;
			this.expiration = exp;
			V v = this.getValue();
			switch (rt) {
			case WEAK:
				if (!(this.ref instanceof ExpirableValuesPseudoMap<?,?,?>.ValueHolder.WeakValueHolderRef)) {
					this.ref = new WeakValueHolderRef(v);
				}
				break;
			case SOFT:
				if (!(this.ref instanceof ExpirableValuesPseudoMap<?,?,?>.ValueHolder.SoftValueHolderRef)) {
					this.ref = new SoftValueHolderRef(v);
				}
				break;
			case HARD:
				if (this.ref!=null) {
					this.ref = null;
				}
				this.value = v;
				break;
			}

			if (this.ttl>0) {
				// if we have a ttl, we need to keep with strong ref
				this.untouch = System.currentTimeMillis()+this.ttl;
				if (this.value==null) {
					this.value = v;
				}
			} else {
				this.untouch = Long.MAX_VALUE;
				// otherwise we do not keep a strong ref, if we have another ref
				// already (i.e, we are not in the HARD case.
				if (this.ref!=null) {
					this.value=null;
				}
			}
			// add to timely if needed.
			this.enqueue();
			return this;
		}

		public V getValue() {
			if (this.value!=null)
				return this.value;
			if (this.ref!=null)
				return this.ref.get();
			return null;
		}

		public K getKey() {
			return this.key;
		}

		public U getExtra() {
			return extra;
		}

		public void setExtra(U extra) {
			this.extra = extra;
		}

		private long nextEvent() {
			return (this.untouch<this.expiration? this.untouch:this.expiration);
		}

		// list timed set, if timely action required, 
		// unlist otherwise
		private void enqueue() {
			if (!timed && (this.expiration<Long.MAX_VALUE) || (this.untouch<Long.MAX_VALUE)) {
				timedValueHolders.add(this);
				timed=true;
			}
		}

		private void dequeue() {
			if (timed) {
				timedValueHolders.remove(this);
				timed=false;
			}
		}

		private void tick(long now) {
			if (now>=this.untouch) {
				// remove if timely
				dequeue();
				this.value = null;
				// reset untouch
				this.untouch=Long.MAX_VALUE;
			}
			if (now>=this.expiration) {
				// remove if timely
				dequeue();
				// list in expiration queue
				if (ExpirableValuesPseudoMap.this.expirationQueue!=null) {
					ExpirableValuesPseudoMap.this.expirationQueue.add(this);
				}
				// reset expiration
				this.expiration=Long.MAX_VALUE;
			}
			// add again, if still needed
			enqueue();
		}

		public long getTtl() {
			return ttl;
		}

		public long getExpiration() {
			return expiration;
		}

		public short getRefMode() {
			if (this.ref==null) {
				return HARD;
			}
			if (this.ref instanceof ExpirableValuesPseudoMap<?,?,?>.ValueHolder.WeakValueHolderRef) {
				return WEAK;
			}
			if (this.ref instanceof ExpirableValuesPseudoMap<?,?,?>.ValueHolder.SoftValueHolderRef) {
				return SOFT;
			}
			throw new IllegalStateException("invalid ref assignment");
		}
	}

	/// END ValueHolder

	private HashMap<K, ValueHolder> backingmap;
	private Map<K, ValueHolder> immutableBackingMapView;
	// external expiration and collection queue. 
	private Queue<ValueHolder> expirationQueue, collectedQueue;

	public ExpirableValuesPseudoMap() {
		this.backingmap = new HashMap<K, ValueHolder>();
		this.immutableBackingMapView = Collections.unmodifiableMap(this.backingmap);
	}

	public ExpirableValuesPseudoMap(int initialSize) {
		this.backingmap = new HashMap<K, ValueHolder>(initialSize);
		this.immutableBackingMapView = Collections.unmodifiableMap(this.backingmap);
	}

	public Queue<ValueHolder> setExpirationQueue(Queue<ValueHolder> q) {
		Queue<ValueHolder> r = this.expirationQueue;
		this.expirationQueue = q;
		return r;
	}

	public Queue<ValueHolder> setInvalidationQueue(Queue<ValueHolder> q) {
		Queue<ValueHolder> r = this.collectedQueue;
		this.collectedQueue = q;
		return r;
	}

	public Map<K,ValueHolder> map() {
		return this.immutableBackingMapView;
	}


	//
	// our special modifying methods
	//
	public ValueHolder  put(K key, V value) {
		if (key==null) {
			throw new NullPointerException("Resource key cannot be null");
		}
		this.countDown--;
		tick();
		ValueHolder vh = new ValueHolder(key,value);
		this.backingmap.put(key, vh);
		int l = this.backingmap.size();
		if (l>this.maxSize)
			this.maxSize = l;
		if (l>this.intMaxSize)
			this.intMaxSize = l; // for shrinks
		return vh;
	}

	public V remove(K key) {
		this.countDown--;
		tick();
		ValueHolder vh = this.backingmap.remove(key);
		if (vh!=null) {
			vh.dequeue();
			return vh.getValue();
		}
		return null;
	}

	public V get(K key) {
		ValueHolder vh = this.backingmap.get(key);
		if (vh!=null)
			return vh.getValue();
		return null;
	}

	public void setTickTimeDelta(int n) {
		this.delta = n;
		this.nextTime = System.currentTimeMillis();
	}

	public void setTickOpCount(int n){
		this.opcount = n;
	}
	//
	// upon a tick we do expirations and cleanouts. This map implementation needs external ticks to do its work
	//

	private int delta = 500;
	private int opcount = 500;
	private long nextTime = System.currentTimeMillis()+delta;
	private int  countDown = opcount;

	@SuppressWarnings("unchecked")
	public void tick() {
		// process ref queue
		Reference<? extends V> r;
		ValueHolder vh;
		while ((r=this.gcQueue.poll())!=null) {
			if (r instanceof ExpirableValuesPseudoMap<?,?,?>.ValueHolder.WeakValueHolderRef) {
				vh = ((ValueHolder.WeakValueHolderRef) r).getValueHolder();
			} else
			if (r instanceof ExpirableValuesPseudoMap<?,?,?>.ValueHolder.SoftValueHolderRef) {
				vh = ((ValueHolder.SoftValueHolderRef) r).getValueHolder();
			} else
				throw new IllegalStateException("internal error: unknown ref type in queue");
			vh.dequeue();
			if (this.collectedQueue!=null) {
				// remove from map must be done in external ticker!
				this.collectedQueue.add(vh);
			} else {
				this.remove(vh.getKey());
			}
		}
		if ((countDown<=0) || (nextTime<=System.currentTimeMillis())) {
			// process timed holders
			// collect first.
			long now = System.currentTimeMillis();
			// Collect those that are due
			List<ValueHolder> due = new LinkedList<>();
			ValueHolder c;
			Iterator<ValueHolder> it = this.timedValueHolders.iterator();
			while (it.hasNext() && (c = it.next()).nextEvent()<=now) {
				due.add(c);
			}
			// tick them all - that should also, if there is nothing left,
			// remove them from the timed value holders.
			due.forEach(v->v.tick(now));
			
			countDown = opcount;
			nextTime  = System.currentTimeMillis()+delta;

			int l = this.backingmap.size();
			int i=1; while ((1<<i)<l) i++;
			int j=1; while ((1<<j)<intMaxSize) j++;
			if ((j-i)>=2) {
				// shrink
				HashMap<K, ValueHolder> nm = new HashMap<K, ValueHolder>(l);
				nm.putAll(this.backingmap);
				// redefine maps
				this.backingmap = nm;
				this.immutableBackingMapView = Collections.unmodifiableMap(this.backingmap);
				this.shrinks++;
				this.intMaxSize=l;
			}
		}
	}

}
