/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.util.sync;



/**
 * A Synchronizer participates in system synchoronization. During a synchronization, a synchronizer may
 * declare any number of resources invalid and subsequently perform synchronization post processing. 
 * <p>
 * The synchronization implementation looks up all synchronizers and implements the following steps:
 * <ol>
 * <li><em>Pre-invalidation:</em> Ask all synchronizers for invalid resources via the {@link #preInvalidation(ISynchronization)} method</li>
 * <li>Invalidate all declared resources.</li>
 * <li><em>Completion:</em>: Complete synchronization via the {@link #complete(ISynchronization)} method.</li> 
 * </ol>
 * 
 * Synchronizers have a priority. Synchronizers with higher priority will be called later during the 
 * pre-invalidation phase and they will be called earlier during the completion phase.
 * <p>
 * The following table lists priorities of common synchronizers:
 * <table border="1" cellspacing="0" cellpadding="5">
 * <tr><th>Synchronizer</th><th>priority</th><th>preInvalidation</th><th>order</th><th>Completion</th><th>order</th></tr>
 * <tr><td>com.zfabrik.boot.repos/synchronizer</td><td>900</td><td>run pre-invalidation on all extensions of AbstractComponentRepo</td><td>run complete on all extensions of AbstractComponentRepo</td></tr>
 * <tr><td>com.zfabrik.components.workeradapter/workerSynchronizer</td><td>800</td><td>&nbsp;</td><td>Reset worker side db</td></tr>
 * <tr><td>com.zfabrik.components/querySynchronizer</td><td>550</td><td>&nbsp;</td><td>Verify component query subscriptions at home and workers</td></tr>
 * <tr><td>com.zfabrik.workers/workerSynchronizer</td><td>500</td><td>&nbsp;</td><td>Propagate invalidations to worker. Attain worker target states</td></tr>
 * <tr><td>com.zfabrik.hubcr/synchronizer</td><td>100</td><td>&nbsp;</td><td>Trigger update scan on worker-side Hub CR implementation</td></tr>
 * <tr><td>com.zfabrik.boot.main/homeSynchronizer</td><td>1</td><td>&nbsp;</td><td>Attain home states (home_up)</td></tr>
 * </table>
 * 
 * @author hb
 *
 */
public interface ISynchronizer {
	/**
	 * Component type of a synchronizer
	 */
	static final String TYPE = "com.zfabrik.synchronizer";
	/**
	 * Component property defining the priority of a synchronizer
	 */
	static final String PRIO = "com.zfabrik.synchronizer.prio";
	
	/**
	 * synchronizers implement this method to collect the set of components to invalidate
	 * and log messages to the end user.
	 * 
	 */
	void preInvalidation(ISynchronization sync);
	
	/**
	 * synchronizers implement this method to re-establish any desired
	 * system state after all invalidations have been performed. Note that the set of synchronizer
	 * in this call may be different to before, due to new system content.
	 */
	void complete(ISynchronization sync);
	
}
