/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.synchronization;

import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.Optional;
import java.util.Set;
import java.util.logging.Logger;

import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.IDependencyComponent;
import com.zfabrik.util.runtime.Foundation;
import com.zfabrik.util.sync.ISynchronization;
import com.zfabrik.util.sync.ISynchronizer;

/**
 * The main synchronizer makes sure that the home process attains its target states initially and
 * at the end of any synchronization
 */
public class HomeSynchronizer implements ISynchronizer, Runnable {
	private final static Logger LOG = Logger.getLogger(HomeSynchronizer.class.getName());
	/**
	 * Hard coded system state that the home process will always attempt to attain first
	 */
	public static final String STATE_HOME_UP 		= "com.zfabrik.boot.main/home_up";

	public void preInvalidation(ISynchronization sync) {}

	public void complete(ISynchronization sync) {
		run();
	}
	
	@Override
	public void run() {
		// Worker processes will re-establish worker_up as a side effect of the worker soul
		if (!Foundation.isWorker()) {
			// this is indeed the home process.
			// Collect states to attain in order
			Set<String> targetStates = new LinkedHashSet<>();
			targetStates.add(STATE_HOME_UP);
			
			// check all compatibility target states from the home layout setting (#1981)
			Optional.ofNullable(
				(String) Foundation.getProperties().get(Foundation.HOME_LAYOUT_COMPONENT)
			)
			.ifPresent(s->targetStates.addAll(Arrays.asList(s.split("\\s*,\\s*"))));
			Optional.ofNullable(
				(String) Foundation.getProperties().get(Foundation.HOME_START)
			)
			.ifPresent(s->targetStates.addAll(Arrays.asList(s.split("\\s*,\\s*"))));
			
			LOG.info("Attaining home process system states ("+String.join(",", targetStates)+")");
			
			for (String targetState:targetStates ) {
				IDependencyComponent st = IComponentsLookup.INSTANCE.lookup(targetState, IDependencyComponent.class);
				if (st==null) {
					LOG.warning("Failed to lookup dependency component (will not be operational): "+st);
				} else {
					st.prepare();
				}
			}
		}
	}
	

}
