/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.fscr;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.logging.Logger;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.components.IDependencyComponent;
import com.zfabrik.components.provider.IComponentsRepository;
import com.zfabrik.components.provider.fs.AbstractFileSystemComponentRepository;
import com.zfabrik.components.provider.fs.FileSystemImpl;
import com.zfabrik.components.provider.fs.IAbstractFileSystem;
import com.zfabrik.resources.ResourceBusyException;
import com.zfabrik.resources.provider.Resource;
import com.zfabrik.util.runtime.Foundation;

/**
* Implementation of a simple File System based repository. As there is not really
* a wealth of information provided and since there is no defined versioning to make use of
* this repository is really simple:
* <ol>
* <li>During a synchronization, a full scan (with configurable depth) will determine the latest
* version. A full scan is computed at startup as well.</li>
* <li>Resources will be downloaded at time of need. Changes on the back end will be effective, independently
* on when the last synchronization was run. </li>
* </ol>
*
* Configuration properties:
* <table border="1" cellpadding="0">
* <tr><td>fscr.checkDepth</td><td>Component folder traversal depth when determining the latest time stamp. Set to less than zero for infinite depth. Default is -1.</td></tr>
* <tr><td>fscr.priority</td><td>Component repository priority. See {@link IComponentsRepository}. Default is 250.</td></tr>
* <tr><td>fscr.folder</td><td>Store folder, i.e. the file system folder that holds the actual resources to run the repository over.</td></tr>
* <tr><td>fscr.base</td><td>Symbolic name of folder the <code>fscr.folder</code> value is evaluated in relation to, if it is a relative folder specification. Legal values are <b>home</b> (default) and <b>here</b>. When set to <b>home</b> the z2 home folder will be considered, if set to <b>here</b> the repository component resource folder will be considered.</td></tr>
* </table>
*
*/
public class FileSystemComponentRepositoryResource extends Resource implements IDependencyComponent {
	private final static Logger LOG = Logger.getLogger(FileSystemComponentRepositoryResource.class.getName());
	private static final String REPO_FOLDER = "fscr.folder";
	private static final String CHECK_DEPTH = "fscr.checkDepth";
	private static final String DEF_PRIO = "250";
	private static final String PRIORITY = "fscr.priority";
    private static final String REPO_BASE = "fscr.base";
    private static final String BASE_HOME = "home";
    private static final String BASE_HERE = "here";


	private AbstractFileSystemComponentRepository fscr;
	private String name;
	private File sroot;

	public FileSystemComponentRepositoryResource(String name) {
		this.name = name;
	}

	@Override
	public synchronized <T> T as(Class<T> clz) {
		if (AbstractFileSystemComponentRepository.has(clz)) {
			_load();
			return fscr.as(clz);
		}
		if (IDependencyComponent.class.equals(clz)) {
			return clz.cast(this);
		}
		return null;
	}

	@Override
	public synchronized void invalidate() throws ResourceBusyException {
		if (this.fscr!=null) {
			try {
				this.fscr.stop();
			} finally {
				this.fscr = null;
			}
		}
	}

	@Override
	public synchronized void prepare() {
		_load();
	}

	@SuppressWarnings("serial")
	private void _load() {
		if (this.fscr==null) {
			IComponentDescriptor d = handle().as(IComponentDescriptor.class);
			//
			// read config
			//
			int prio;
			try {
				prio = Integer.parseInt(d.getProperties().getProperty(PRIORITY,DEF_PRIO).trim());
			} catch (NumberFormatException nfe) {
				throw new IllegalArgumentException("Bad specification of "+PRIORITY+": "+name,nfe);
			}
			int checkDepth;
			try {
				checkDepth = Integer.parseInt(d.getProperties().getProperty(CHECK_DEPTH,"-1").trim());
			} catch (NumberFormatException nfe) {
				throw new IllegalArgumentException("Bad specification of "+CHECK_DEPTH+": "+name,nfe);
			}
			if (checkDepth<0) {
				// really deep... but only half of max to avoid overflows.
				checkDepth=Integer.MAX_VALUE >> 1;
			}
			String ssroot = normalize(d.getProperty(REPO_FOLDER));
			if (ssroot == null) {
				throw new IllegalArgumentException("Missing specification of "+REPO_FOLDER+": "+name);
			}
			this.sroot = new File(ssroot);
			if (!this.sroot.isAbsolute()) {
				// relative path
                String base = d.getProperties().getProperty(REPO_BASE,BASE_HOME).trim();
                if (BASE_HERE.equals(base)) {
					try {
						this.sroot = new File(IComponentsManager.INSTANCE.retrieve(this.name), ssroot);
					} catch (IOException ioe) {
						throw new IllegalStateException("Component " + this.name + " not resolved by CM", ioe);
					}
                } else {
                    if (BASE_HOME.equals(base)) {
                        this.sroot = new File(Foundation.getHomeFolder(),ssroot);
                    } else {
                        throw new IllegalArgumentException("Value "+base+" for "+REPO_BASE+" not supported");
                    }
                }
        	}
			if (!this.sroot.exists()) {
				throw new IllegalArgumentException("File system storage at "+this.sroot.getAbsolutePath()+" not found!");
			}
			this.fscr = new AbstractFileSystemComponentRepository(
				this.name,
				prio,
				checkDepth
			);
			// set root
			this.fscr.setRoots(new HashMap<String,IAbstractFileSystem>(){{ put("",new FileSystemImpl(sroot)); }});
			this.fscr.start();
			handle().attach(this.fscr);
			
			if (!Foundation.isWorker()) {
				LOG.info("Using FS-CR: " + this.fscr.toString());
			}
		}
	}


	private String normalize(String in) {
		return 	(in==null? 	null : (((in=in.trim()).length()==0)?	null :	in));
	}

}
