/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.components.java;

import static com.zfabrik.components.IComponentDescriptor.COMPONENT_TYPE;
import static com.zfabrik.components.java.build.ICompiler.COMPILER_ID;
import static com.zfabrik.util.expression.X.val;
import static com.zfabrik.util.expression.X.var;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.concurrent.Callable;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.components.java.build.ICompiler;
import com.zfabrik.resources.IResourceHandle;
import com.zfabrik.util.expression.X;
import com.zfabrik.util.html.Escaper;
import com.zfabrik.util.runtime.Foundation;
import com.zfabrik.util.threading.ThreadUtil;


/**
 * Utility functions around Java components. These are useful when implementing a component type that requires to load Java classes as well 
 * as anything that consumes Java component names (and should use the defaulting methods).
 * 
 * @author hb
 *
 */
public class JavaComponentUtil {


	/**
	 * Given a component name return the default Java component for it. That is, a project name will be completed
	 * by /java, a component name will be appended /java after removing the last name segment
	 *
	 * @param forComponent A component name to derive a Java component lookup from
	 * @return The resource handle of a derived Java component
	 */
	public static IResourceHandle getJavaComponent(String forComponent) {
		IResourceHandle rh = IComponentsLookup.INSTANCE.lookup(getJavaComponentName(forComponent),IResourceHandle.class);
		return rh;		
	}
	

	/**
	 * Given a component name return the default Java component name for it. That is, a project name will be completed
	 * by /java, a component name will be appended /java after removing the last name segment
	 *
	 * @param forComponent A component name to derive a Java component name from
	 * @return The name of a derived Java component
	 */
	public static String getJavaComponentName(String forComponent) {
		int p = forComponent.lastIndexOf('/');
		if (p<0) {
			return forComponent+"/java";
		} 
		return forComponent.substring(0,p)+"/java";
	}

	/**
	 * Given a module name or a component name, determine a java component name.  
	 * This is not about the default java component name. Use {@link #getJavaComponentName(String)} for that.
	 * This does not alter a fully qualified component name, but appends /java to a module name
	 * 
	 * @param forModuleOrComponent The name of a Java component or a module
	 * @return A Java component name
	 */
	public static String fixJavaComponentName(String forModuleOrComponent) {
		return (forModuleOrComponent.lastIndexOf('/')<0?forModuleOrComponent+"/java":forModuleOrComponent);
	}
	
	/**
	 * Load an implementation for a component by looking up the associated Java component
	 * In addition, manage the dependency to the Java component
	 * 
	 * @param componentName A component name
	 * @param clzNameProp A property declaring a class name in the component's descriptor
	 * @param r A resource handle that will be dependent on the loading Java component
	 * @param type The expected implementation type of the implementation class
	 * @return the implementation instance
	 */
	public static <T> T loadImplementationFromJavaComponent(String componentName, String clzNameProp, IResourceHandle r, Class<T> type) {
		synchronized (r) {
			IComponentDescriptor desc = IComponentsManager.INSTANCE.getComponent(componentName);
			if (desc == null) 
				throw new IllegalArgumentException("component not found: "+componentName);
			if (desc.getProperties()==null)
				throw new IllegalStateException("component has no properties: "+componentName);
			String clzName = desc.getProperties().getProperty(clzNameProp);
			if (clzName==null)
				throw new IllegalStateException("component property \""+clzNameProp+"\" missing: "+componentName);
			IResourceHandle rh = getJavaComponent(componentName);
			r.addDependency(rh);    				
			IJavaComponent jc = rh.as(IJavaComponent.class);
			if (jc==null)
				throw new IllegalStateException("failed to retrieve java component for component: "+componentName);
			try {
				return type.cast(Class.forName(clzName,false,jc.getPrivateLoader()).getConstructor().newInstance());
			} catch (LinkageError le) {
				throw new IllegalStateException("failed to load implementation of component: "+componentName,le);
			} catch (Exception e) {
				throw new IllegalStateException("failed to load implementation of component: "+componentName,e);
			}
		}
	}

	/**
	 * load an implementation class for a component by looking up the associated Java component
	 * In addition, manage the dependency to the Java component
	 * 
	 * @param componentName A component name
	 * @param clzNameProp A property declaring a class name in the component's descriptor
	 * @param r A resource handle that will be dependent on the loading Java component
	 * @return the implementation class
	 */
	public static Class<?> loadImplementationClassFromJavaComponent(String componentName, String clzNameProp, IResourceHandle r) {
		synchronized (r) {
			IComponentDescriptor desc = IComponentsManager.INSTANCE.getComponent(componentName);
			if (desc == null) 
				throw new IllegalArgumentException("component not found: "+componentName);
			if (desc.getProperties()==null)
				throw new IllegalStateException("component has no properties: "+componentName);
			String clzName = desc.getProperties().getProperty(clzNameProp);
			if (clzName==null)
				throw new IllegalStateException("component property \""+clzNameProp+"\" missing: "+componentName);
			IResourceHandle rh = getJavaComponent(componentName);
			r.addDependency(rh);    				
			IJavaComponent jc = rh.as(IJavaComponent.class);
			if (jc==null)
				throw new IllegalStateException("failed to retrieve java component for component: "+componentName);
			try {
				return Class.forName(clzName,false,jc.getPrivateLoader());
			} catch (LinkageError le) {
				throw new IllegalStateException("failed to load implementation of component: "+componentName,le);
			} catch (Exception e) {
				throw new IllegalStateException("failed to load implementation of component: "+componentName,e);
			}
		}
	}
	
	/**
	 * Gets the private class loader of the default java component for the given component name. Defaults to
	 * the passed default loader, if no Java component is defined.
	 */
	public static ClassLoader getPrivateLoaderOfJavaComponent(String componentName, ClassLoader defaultLoader) {
		IResourceHandle rh = getJavaComponent(componentName);
		IJavaComponent jc = rh.as(IJavaComponent.class);
		if (jc==null) {
			return defaultLoader;
		}
		return jc.getPrivateLoader();
	}
	
	/**
	 * Gets the private class loader of the default java component for the given component name. Defaults to
	 * the core class loader
	 */
	public static ClassLoader getPrivateLoaderOfJavaComponent(String componentName) {
		return getPrivateLoaderOfJavaComponent(componentName,JavaComponentUtil.class.getClassLoader());
	}

	/**
	 * Gets a java compiler implementation component by id. This is a utility function that simply issues the
	 * corresponding query and checks for existence and uniqueness.
	 * 
	 * @param compilerID The id of a compiler
	 * @return The component name providing the compiler
	 * @throws IOException
	 */
	public static String getCompilerComponentById(String compilerID) throws IOException {
		X query = 
				var(COMPONENT_TYPE).eq(val(ICompiler.TYPE))
			.and(
				var(COMPILER_ID).eq(val(compilerID))
			);

		Collection<String> found = IComponentsManager.INSTANCE.findComponents(query);

		// there can be only one!
		if (found.size() == 0) {
			throw new UnsupportedOperationException("No compiler support found for "	+ compilerID);
		}
		if (found.size() > 1) {
			String ambiguous = "";
			for (String c : found) {
				ambiguous += c + " ";
			}
			throw new IllegalStateException("More than one compiler found for "+ compilerID + ": " + ambiguous);
		}

		// get the only one!
		String compilerComponentName = found.iterator().next();
		return compilerComponentName;
	}
	
	/**
	 * Creates a context class loader switching invocation wrapper by passing an
	 * interface, an implementation, and a component name.
	 */
	public static <T> T privateContextWrap(String cn, final T t, final Class<T> intf) {
		if (t==null) {
			return null;
		}
		final ClassLoader pl = getPrivateLoaderOfJavaComponent(cn);
		return intf.cast(Proxy.newProxyInstance(pl, new Class[]{intf}, new InvocationHandler() {
			@Override
			public Object invoke(Object proxy, final Method method, final Object[] args) throws Throwable {
				return ThreadUtil.cleanContextExceptionExecute(new Callable<Object>() {
					@Override
					public Object call() throws Exception {
						return method.invoke(t, args);
					}
				});
			}
		}));
	}
	
	/**
	 * Like {@link ThreadUtil#cleanContextExceptionExecute(ClassLoader, Callable)} but simply based on a component identification
	 */
	public static <T> T privateContextExceptionExecute(String componentName,Callable<T> callable) throws Exception {
		return ThreadUtil.cleanContextExceptionExecute(
			getPrivateLoaderOfJavaComponent(componentName),
			callable
		);
	}
	
	/**
	 * Like {@link ThreadUtil#cleanContextExceptionExecute(ClassLoader, Callable)} but simply based on a component identification
	 */
	public static <T> T privateContextExecute(String componentName,Callable<T> callable) {
		return ThreadUtil.cleanContextExecute(
			getPrivateLoaderOfJavaComponent(componentName),
			callable
		);
	}
	
	/**
	 * Parse a reference and includes list and normalize component names.
	 */
	public static Collection<String> parseDependencies(String s) {
		return _parseDependencies(s, true);
	}
	
	
	public static Collection<String> _parseDependencies(String s,boolean applyQuery) {
		// we want to preserve order of references as listed
		Set<String> deps = new LinkedHashSet<String>();
		if (s!=null) {
			StringTokenizer tk = new StringTokenizer(s,",");
			while (tk.hasMoreTokens()) {
				String n = tk.nextToken().trim();
				if (n.length()>0) {
					String m = JavaComponentUtil.fixJavaComponentName(stripQuery(n));
					if (applyQuery) {
						Map<String,String> q = extractQuery(n);
						String x = q.get("expand");
						if (x!=null && Boolean.valueOf(x)) {
							deps.addAll(expand(m));
						}
					}
					deps.add(m);
				}
			}
		}
		return deps;
	}
	
	
	//
	// refs may be modified to be transitive. We resolve that here in a depth-last way
	//
	private static Collection<String> expand(String expand) {
		Set<String> don = new HashSet<String>();
		// preserve order
		Set<String> res = new LinkedHashSet<String>();
		Collection<String> work = new HashSet<String>();
		work.add(expand);
		while (!work.isEmpty()) {
			Set<String> add = new HashSet<String>();
			for (String inc : work) {
				if (!don.contains(inc)) {
					don.add(inc);
					// expand
					IComponentDescriptor d = IComponentsManager.INSTANCE.getComponent(inc);
					if (d!=null) {
						add.addAll(_parseDependencies(d.getProperty(IJavaComponent.PUBREFS),false));
					}
					res.add(inc);
				}
			}
			work = add;
		}
		return res;
	}

	// strip off any query
	private static String stripQuery(String n) {
		int p = n.lastIndexOf('?');
		if (p>0) {
			return n.substring(0,p);
		}
		return n;
	}

	// extract and parse query
	private static Map<String,String> extractQuery(String q) {
		if (q!=null) {
			int p = q.lastIndexOf('?');
			if (p>=0) {
				StringTokenizer tk = new StringTokenizer(q.substring(p+1), "&");
				// preserve order
				Map<String,String> m = new LinkedHashMap<String, String>();
				while (tk.hasMoreTokens()) {
					String nv = tk.nextToken().trim();
					int s = nv.lastIndexOf('=');
					if (s>0) {
						m.put(
							Escaper.urlDecode(nv.substring(0,s)).trim(),
							Escaper.urlDecode(nv.substring(s+1)).trim()
						);
					}
				}
				return m;
			}
		}
		return Collections.emptyMap();
	}


	/**
	 * Given a Java component descriptor, check, if no build is required
	 */
	public static boolean isPreBuilt(IComponentDescriptor desc) {
		return Boolean.parseBoolean(desc.getProperties().getProperty(IJavaComponent.NOBUILD, "false"));
	}


	/**
	 * Given a component folder, compute the instance folder. The instance
	 * folder is where after initialization, we find the runtime resources
	 * of the Java component, including build artifacts.
	 */
	public static File getInstanceFolder(File componentFolder) {
		String name = componentFolder.getName();
		// make sure to remove any instance indication
		int p = name.lastIndexOf('_');
		if (p>0) {
			name = name.substring(0,p);
		}
		return new File(componentFolder.getParentFile(),name+"_"+Foundation.getInstanceId());
	}
	


}
