/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.home.console.gui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.management.ManagementFactory;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JToolBar;
import javax.swing.ListSelectionModel;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellRenderer;

import com.zfabrik.home.console.gui.ManagementConsole.ConsoleRunner;

@SuppressWarnings("serial")
public class WorkersPane extends AbstractTableModel implements ActionListener {
	private static final String AC_ALL  = "MA";
	private static final String AC_NONE = "MN";
	private static final String AC_PING = "PI";
	private static final String AC_STOP = "ST";
	private final static DateFormat df = DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM);
	private static final String[] COLS = new String[] { "mark", "name", "started", "status", "ping"};
	private ObjectName PATTERN;
	private List<WorkerRow> workers = new ArrayList<WorkerRow>();
	private MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
	private JComponent root;
	private JTable table;
	private ConsoleRunner runner;
	
	// construction
	public WorkersPane(ConsoleRunner runner) {
		this.runner = runner;
		try {
			PATTERN = ObjectName.getInstance("zfabrik:type=com.zfabrik.impl.workers.home.WorkerProcessResource,name=*");
		} catch (Exception e) {
			throw new IllegalStateException("Failed to start worker panel", e);
		}
	}

	public JComponent getComponent() {
		if (root == null) {
			JPanel content = new JPanel(new BorderLayout());
			
			JToolBar actions = new JToolBar();
			
			JButton  bu = new JButton("all",runner.imageIcon("/icons/hb_select_all.png"));
			bu.setActionCommand(AC_ALL);
			bu.addActionListener(this);
			actions.add(bu);

			bu = new JButton("none",runner.imageIcon("/icons/hb_select_none.png"));
			bu.setActionCommand(AC_NONE);
			bu.addActionListener(this);
			actions.add(bu);
			
			bu = new JButton("ping",runner.imageIcon("/icons/server_lightning.png"));
			bu.setActionCommand(AC_PING);
			bu.addActionListener(this);
			actions.add(bu);

			bu = new JButton("stop",runner.imageIcon("/icons/server_delete.png"));
			bu.setActionCommand(AC_STOP);
			bu.addActionListener(this);
			actions.add(bu);
			
			content.add(actions,BorderLayout.NORTH);
			
			table = new JTable(this) {
				@Override
				public Component prepareRenderer(TableCellRenderer renderer,int row, int column) {
					Component c = super.prepareRenderer(renderer, row, column);
					if (c instanceof JCheckBox) {
						// scale it here somehow
						JCheckBox jcb = (JCheckBox) c;
						Dimension d = jcb.getPreferredSize();
						jcb.setSize(
							(int) (d.getWidth()*runner.getScale()), 
							(int) (d.getHeight()*runner.getScale())
						);
					}
					return c;
				}	
			};
			table.setFillsViewportHeight(true);
			table.setRowHeight((int) (table.getRowHeight()*runner.getScale()));
			table.getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
			JScrollPane sp = new JScrollPane(table);
			content.add(sp,BorderLayout.CENTER);

			// mouse wheel based zoom
			sp.addMouseWheelListener((e)->{			
		       if (e.isControlDown()) {
		    	   int notches = e.getWheelRotation();
		    	   if (notches>0) {
		    		   zoomOut();
		    	   }
		    	   if (notches<0) {
		    		   zoomIn();
		    	   }
		        }
			});
			this.root = content;
		}
		return root;
	}

	//

	private class WorkerRow {
		private static final String N_A = "n.a";
		boolean marked;
		ObjectName on;
		Date started;
		Short status;
		long ping = -1;
		String error;

		public WorkerRow(ObjectName on) {
			try {
				this.on = on;
				refresh();
			} catch (Exception e) {
				throw new IllegalStateException("Failed to setup worker record",e);
			}
		}
		public void refresh() {
			try {
				this.started = (Date) mbs.getAttribute(on, "LastSuccessfulStart");
				Throwable err = (Throwable) mbs.getAttribute(on, "LastError");
				if (err!=null) {
					this.error = err.toString();
				} else {
					this.error = null;
				}
				this.status = (Short) mbs.getAttribute(on,"State");
			} catch (Exception e) {
				throw new IllegalStateException("Failed to refresh worker data",e);
			}
		}

		public Object getColumn(int columnIndex) {
			switch (columnIndex) {
			case 0:
				return this.marked;
			case 1:
				String pid; 
				try {
					pid = (String) mbs.getAttribute(on, "PID");
				} catch (Exception e) {
					pid = null;
				}
				try {
					return (String) mbs.getAttribute(on, "Name") + (pid!=null? " ("+pid+")":"");
				} catch (Exception e) {
					return e.toString();
				}
			case 2:
				return (this.started!=null? df.format(this.started) : N_A);
			case 3:
				String st = N_A;
				if (this.status!=null) {
					switch (this.status.shortValue()) {
					case 0: st="not started"; break;
					case 1: st="starting"; break;
					case 2: st="started"; break;
					case 3: st="detached"; break;
					case 4: st="stopping"; break;
					}
				}
				if (this.error!=null) {
					return st+" (error:"+this.error+")";
				}
				return st;
			case 4:
				return this.ping+"ms";
			default:
				return null;
			}
		}
		public void ping() {
			try {
				this.ping = (Long) mbs.invoke(this.on,"ping",null,null);
			} catch (Exception e) {
				error("Action Error","Ping failed: "+e.toString());
			}
		}

		public void stop() {
			// make sure, we don't go into the worker list refresh while
			// stopping a worker (block's ui)
			// fork a thread for this
			final MBeanServer m = mbs;
			final ObjectName o = this.on;
			new Thread(new Runnable() {
				public void run() {
					boolean prev = runner.refreshWorkers;
					try {
						runner.refreshWorkers=false;
						m.invoke(o, "stop", null, null);
					} catch (InstanceNotFoundException infe) {
						// ignore
					} catch (Exception e) {
						error("Action Error", "Stop failed: " + e.toString());
					} finally {
						runner.refreshWorkers=prev;
					}
				}
			}).start();
		}

	}

	private void error(String caption, String msg) {
		JOptionPane.showMessageDialog(this.root, msg, caption, JOptionPane.ERROR_MESSAGE);
	}
	
	public Class<?> getColumnClass(int columnIndex) {
		switch (columnIndex) {
		case 0:
			return Boolean.class;
		case 1:
			return String.class;
		case 2:
		case 3:
			return String.class;
		case 4:
			return String.class;
		default:
			return null;
		}
	}
	
	public boolean isCellEditable(int rowIndex, int columnIndex) {
		return (columnIndex==0);
	}

	public void setValueAt(Object value, int rowIndex, int columnIndex) {
		if (columnIndex==0) {
			WorkerRow w = _getWR(rowIndex);
			if (w!=null) w.marked = ((Boolean) value).booleanValue();
		}
	}
	public String getColumnName(int column) {
		return COLS[column];
	}

	public int getColumnCount() {
		return COLS.length;
	}

	public int getRowCount() {
		return this.workers.size();
	}

	private WorkerRow _getWR(int i) {
		if ((i>=0) && (i<this.workers.size())) {
			return this.workers.get(i);
		}
		return null;		
	}
	
		
	public Object getValueAt(int rowIndex, int columnIndex) {
		WorkerRow r = _getWR(rowIndex);
		if (r!=null)
			return r.getColumn(columnIndex);
		return null;
	}

	public synchronized void refresh() {
		Set<ObjectName> workerObjectNames = mbs.queryNames(PATTERN, null);
		Set<ObjectName> alreadyPresentWorkers = new HashSet<ObjectName>();
		List<WorkerRow> updatedWorkerRowList = new ArrayList<WorkerRow>(workerObjectNames.size());
		for (WorkerRow r : this.workers) {
			if (workerObjectNames.contains(r.on)) {
				updatedWorkerRowList.add(r);
				alreadyPresentWorkers.add(r.on);
			}
		}
		workerObjectNames.removeAll(alreadyPresentWorkers);
		for (ObjectName on : workerObjectNames) {
			updatedWorkerRowList.add(new WorkerRow(on));
		}
		this.workers = updatedWorkerRowList;
		for (WorkerRow r : this.workers) {
			r.refresh();
		}
		invalidate();
	}


	public void actionPerformed(ActionEvent e) {
		String action = e.getActionCommand();
		if (AC_ALL.equals(action)) {
			for (WorkerRow r : this.workers) { r.marked=true; }
			invalidate();
		} else
		if (AC_NONE.equals(action)) {
			for (WorkerRow r : this.workers) { r.marked=false; }
			invalidate();
		} else
		if (AC_PING.equals(action)) {
			int i=0;
			for (WorkerRow r : this.workers) { 
				if (r.marked) {
					r.ping();
					invalidate(i);
				}
				i++;
			}
		} else
		if (AC_STOP.equals(action)) {
			for (WorkerRow r : this.workers) { 
				if (r.marked) {
					r.stop();
				}
			}
			invalidate();
		}
	}

	private void invalidate(int i) {
		int s = this.table.getSelectionModel().getAnchorSelectionIndex();
		this.fireTableRowsUpdated(i,i);
		if (s>=0) this.table.getSelectionModel().setSelectionInterval(s, s);
	}

	private void invalidate() {
		int s = this.table.getSelectionModel().getAnchorSelectionIndex();
		this.fireTableDataChanged();
		if (s>=0) this.table.getSelectionModel().setSelectionInterval(s, s);
		// recompute row heights
	    for (int row = 0; row < table.getRowCount(); row++) {
	        int rowHeight = table.getRowHeight();
	        for (int column = 0; column < table.getColumnCount(); column++) {
	            Component comp = table.prepareRenderer(table.getCellRenderer(row, column), row, column);
	            rowHeight = Math.max(rowHeight, comp.getPreferredSize().height);
	        }
	        table.setRowHeight(row, rowHeight);
	    }
	}
	
	public void zoomIn() {
		if (table.getFont().getSize()<36) { 
			table.setFont(table.getFont().deriveFont((float)(table.getFont().getSize()+1)));
			invalidate();
		}
	}
	
	public void zoomOut() {
		if (table.getFont().getSize()>1) { 
			table.setFont(table.getFont().deriveFont((float)(table.getFont().getSize()-1)));
			invalidate();
		}
	}

}
