/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components;

import static com.zfabrik.impl.components.ComponentDependencyUtil.addDepsFromCSL;
import static com.zfabrik.impl.components.ComponentDependencyUtil.prepareDependencies;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.java.IJavaComponent;
import com.zfabrik.components.provider.IComponentFactory;
import com.zfabrik.components.provider.IComponentsRepository;
import com.zfabrik.impl.components.ComponentDependencyUtil.IDependencyResolver;
import com.zfabrik.impl.components.java.JavaComponentImpl;
import com.zfabrik.resources.IResourceHandle;
import com.zfabrik.resources.ResourceBusyException;
import com.zfabrik.resources.provider.Resource;
import com.zfabrik.util.expression.X;

/**
 * Wraps another resource for the resource management. This is done so that a
 * release of the resource really releases the wrapped resource. I.e. the handle
 * registered with the resource management is not the one provided by the now
 * released resource.
 * <p>
 * This wrapper implements the indirection via component factories.
 * <p>
 * Another relevant reason to have a wrapper is to de-couple the class loading.
 * That is, if a resource consumer holds on to a resource handle, that in turn
 * holds on to a resource, the consumer implicitly holds on to the resource's
 * implementation type class loader.
 * <p>
 * Furthermore the wrapper implements some default behaviours,
 * such as:
 * <ul>
 * <li>Providing an {@link IComponentDescriptor} when asked</li>
 * <li>Providing a dummy resource when no component by the name exists</li>
 * <li>Maintaining a repository dependency to the providing component repository</li>
 * <li>Handling dependencies declared via {@link IComponentDescriptor#DEPENDENCIES}</li>
 * </ul>
 * 
 * @author hb
 * 
 */
public class ComponentResourceWrapper extends Resource {

	private final static Resource DUMMY = new NOPResource();
	private final String name;
	private String type;
	private Resource wrapped;
	private IResourceHandle delegate;
	private IComponentDescriptor desc;

	public ComponentResourceWrapper(String name) {
		this.name = name;
	}

	public <T> T as(Class<T> clz) {
		check();
		// if a delegation is specified, delegate
		if (this.delegate!=null) {
			return this.delegate.as(clz);
		}
		// default only, if there is no delegation.
		// Note: This way looking up a component via the component manager
		// is truly different than looking up via the resource handle
		if (IComponentDescriptor.class.equals(clz)) {
			return clz.cast(fetchDesc());
		}
		if (this.wrapped!=null) {
			return this.wrapped.as(clz);
		}
		return null;
	}

	public void invalidate() throws ResourceBusyException {
		synchronized (this) {
			if (logger.isLoggable(Level.FINE)) {
				logger.fine("Invalidating "+this.name);
			}
			try {
				if (this.wrapped != null) {
					this.wrapped.invalidate();
				}
			} finally {
				this.wrapped=null;
				this.desc = null;
				this.type = null;
				this.delegate=null;
			}
		}
	}

	private IComponentDescriptor fetchDesc() {
		return ComponentsManager.INSTANCE.getComponent(name);
	}
	
	private synchronized void check() {
		if (this.wrapped==null && this.delegate==null) {
			_loadDependencies();
			_load();
		}
	}

	private void _loadDependencies() {
		prepareDependencies(
			handle(),
			new IDependencyResolver() {
				public Collection<String> getDependencies() throws Exception {
					IComponentDescriptor d = fetchDesc();
					if (d==null) {
						return Collections.emptySet();
					}
					Set<String> deps = new HashSet<String>();
					addDepsFromCSL(deps,d.getProperty(IComponentDescriptor.DEPENDENCIES));
					return deps;
				}
			},
			logger
		);
	}

	private void _load() {
		this.desc = fetchDesc();
		if (this.desc == null) {
			// use a dummy handle that does nothing... but you can
			// depend on :-)
			this.wrapped = DUMMY;
		} else {
			// Get its type
			this.type = this.desc.getType();
			
			if (type == null) {
				// #841 default to "any" component
				type = IComponentDescriptor.ANY_COMPONENT_TYPE;
			}
			type=type.trim();
			
			// check for a repo name property and make the component dependent if so.
			String repo = this.desc.getProperties().getProperty(IComponentsRepository.COMPONENT_REPO_IMPLEMENTATION);
			if (repo!=null) {
				handle().addDependency(IComponentsLookup.INSTANCE.lookup(repo, IResourceHandle.class));
			}		
			
			if (IJavaComponent.TYPE.equals(type)) {
				// it's a java component. Must take care of it right
				// here since we use java to bootstrap
				Resource r = new JavaComponentImpl(name);
				r.init(handle());
				this.wrapped = r;
			} else 
			if (IComponentFactory.TYPE.equals(type)) {
				// it is a component factory. We handle that right here
				// (again, otherwise
				// we wouldn't know how to bootstrap.
				Resource r = new ComponentFactoryImpl(name);
				r.init(handle());
				this.wrapped = r;
				// the component factory of a component factory is... the java component (means: dump on invalidation)
			} else
			if (IComponentDescriptor.LINK_COMPONENT_TYPE.equals(type)) {
				// It's a link. So we delegate.
				String target = this.desc.getProperty(IComponentDescriptor.LINK_TARGET_COMPONENT);
				if (target==null || (target=target.trim()).length()==0) {
					throw new IllegalStateException("Missing specification of "+IComponentDescriptor.LINK_TARGET_COMPONENT+": "+name);
				}
				this.delegate=IComponentsLookup.INSTANCE.lookup(target, IResourceHandle.class);
				// add dependency
				handle().addDependency(this.delegate);
			} else {
				if (logger.isLoggable(Level.FINE)) {
					logger.fine("Producing component resource wrapper: "+this.name);
				}
				// get the component factory.
				Collection<String> cs;
				try {
					cs = ComponentsManager.INSTANCE.findComponents(X.and(X.var(IComponentDescriptor.COMPONENT_TYPE).eq(X.val(IComponentFactory.TYPE)),
							X.var(IComponentFactory.IMPLEMENTED_TYPE).eq(X.val(type))));
				} catch (IOException ioe) {
					throw new IllegalStateException("failed to retrieve component factory for type " + type, ioe);
				}
				if (cs.isEmpty())
					throw new IllegalStateException("No component factory found for component "+name +" of type " + type);
				if (cs.size()>1)
					throw new IllegalStateException("More than one component factory found for component "+name +" of type " + type + ": "+cs);

				String cfn = cs.iterator().next();
				IResourceHandle cfh = IComponentsLookup.INSTANCE.lookup(cfn, IResourceHandle.class);
				IComponentFactory cf = cfh.as(IComponentFactory.class);
				if (cf == null)
					throw new IllegalStateException("Failed to retrieve component factory \"" + cfn + "\" for component: " + name);
				Resource r = cf.createComponentResource(name);
				if (r == null)
					throw new IllegalStateException("Component factory \"" + cfn + "\" did not provide component resource: " + name);
				// NOTE: via the context implementation indirection, wrapped keeps a ref on this wrapper - which is 
				// crucial as it binds this wrappers life cycle to the wrapped resource
				r.init(handle());
				// we make the component resource depend on its factory
				// NOTE: Java Component Dependency is individual and a side 
				// effect of using the Java Component Utils!!! That's why
				// it doesn't show up here!
				handle().addDependency(cfh);
				this.wrapped = r;
			}
		}
	}		
	
	private final static Logger logger = Logger.getLogger(ComponentResourceWrapper.class.getName());
}
