/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.java;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.AccessControlContext;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.components.java.IJavaComponent;
import com.zfabrik.components.java.JavaComponentClassLoader;
import com.zfabrik.components.java.JavaComponentUtil;
import com.zfabrik.components.java.build.IJavaBuilder;
import com.zfabrik.resources.IResourceHandle;
import com.zfabrik.resources.ResourceBusyException;
import com.zfabrik.resources.provider.Resource;
import com.zfabrik.util.runtime.Foundation;

/**
 * Constructs Java component representation. Other components may expect a java
 * component by the hard coded (conventional) name "java". There can be more
 * than one though.
 * 
 * @see {@link IJavaComponent}
 * @author hb
 * 
 */
public class JavaComponentImpl extends Resource implements IJavaComponent {

	// class path constants
	private static final String BIN_API = "bin.api";
	private static final String BIN_IMPL = "bin.impl";
	private static final String BIN_TEST = "bin.test";
	
	// alternative of bin.api
	private static final String BIN = "bin";
	// deprecated:
	private static final String BIN_PRIVATE = "bin/private";

	/*
	 * A speed valve to slow JRE loaders
	 */
	private final static ClassLoader PARENT_LOADER = new SpeedValveLoader(Resource.class.getClassLoader());
	
	/*
	 *  count invalidations. Used to check for invalidations in ComponentResourceWrapper
	 */
	private static volatile long COUNTER = 0;
	
	private static AccessControlContext acc = AccessController.getContext();
		
	private IComponentDescriptor desc;
	private String name;
	private boolean inited;
	private long counter = COUNTER;
	private File runtimeResources;

	private ComponentClassLoader privloader, publloader;

	public JavaComponentImpl(String name) {
		this.name = name;
	}

	public <T> T as(Class<T> clz) {
		synchronized (this) {
			if (IJavaComponent.class.equals(clz)) {
				if (!inited) {
					if (logger.isLoggable(Level.FINE)) {
						logger.fine("Loading Java component: "+this.name);
					}
					AccessController.doPrivileged((PrivilegedAction<Object>)
						/*
						 * By executing the initialization in a privileged action, our classloaders
						 * get a fresh access control context. Otherwise, they would hold on to 
						 * potentially several protection domains, which in turn hold on to other
						 * class loaders (traversed on the call stack) that may not be valid anymore (i.e. a leak!)
						 */
						()->{
							try {
								List<IJavaComponent> pubrefs,privrefs;
								
								File origin = IComponentsManager.INSTANCE.retrieve(JavaComponentImpl.this.name);
								
								IComponentDescriptor desc = _getDescriptor();
								boolean noBuild = JavaComponentUtil.isPreBuilt(desc);
								if (noBuild) {
									// for no-build components (only really true for pre-built resources necessary for the bootstrapping process)
									// the runtime resources folder is simply the origin (there is by definition no instance specific modifications 
									// supported)
									JavaComponentImpl.this.runtimeResources = origin;
								} else {
									// the instance root for the java component
									// is a sibling of the of the original resources.
									JavaComponentImpl.this.runtimeResources = JavaComponentUtil.getInstanceFolder(origin);
								}
								
								// read references and set dependencies
								StringTokenizer tokenizer;
								String cn;

								// compiler dependencies
								String compilers = desc.getProperties().getProperty(COMPILE_ORDER,"java");
								tokenizer = new StringTokenizer(compilers,",");
								while (tokenizer.hasMoreTokens()) {
									String n = tokenizer.nextToken().trim();
									if (n.length()>0) {
										cn = JavaComponentUtil.getCompilerComponentById(n);
										handle().addDependency(IComponentsLookup.INSTANCE.lookup(cn, IResourceHandle.class));
									}
								}

								// refs and includes to dependencies
								boolean devMode = Foundation.MODE_DEVELOPMENT.equals(System.getProperty(Foundation.MODE));

								// public refs
								pubrefs = addDependencies(desc.getProperties().getProperty(PUBREFS),false);
								// private and test refs (all end up in the same classloader)
								privrefs = addDependencies(desc.getProperties().getProperty(PRIREFS),false);
								if (devMode) {
									privrefs.addAll(addDependencies(desc.getProperties().getProperty(TESTREFS),false));
								}
								// all include based deps
								addDependencies(desc.getProperty(PUBINCS),true);
								addDependencies(desc.getProperty(PRIINCS),true);
								if (devMode) {
									addDependencies(desc.getProperty(TESTINCS),true);
								}

								// could be just properties (refs) and no code.
								if (runtimeResources!=null && !noBuild) {
									// check make (unless it has the NO MAKE flag)
									IJavaBuilder.INSTANCE.make(desc, origin, runtimeResources, logger);
								}
								
								setupClassLoaders(pubrefs, privrefs, runtimeResources, desc, devMode);

								// done
								JavaComponentImpl.this.inited = true;
							} catch (Exception e) {
								_clean();
								throw new IllegalStateException("failed to initialize java component: " + JavaComponentImpl.this.name, e);
							}
							return null;
						},
						acc
					);
					if (logger.isLoggable(Level.FINE)) {
						logger.fine("Java component loaded: "+this.name);
					}
				}
				return clz.cast(this);
			}  else 
			if (clz.equals(Long.class)) {
				// access to invalidation counter
				return clz.cast(counter);
			}
		}
		return null;
	}

	/*
	 * check refs or includes and add to dependencies.
	 */
	private List<IJavaComponent> addDependencies(String refsOrIncludes, boolean includes) {
		List<IJavaComponent> javaDeps = new LinkedList<IJavaComponent>();
		if (refsOrIncludes != null) {
			Collection<String> deps = JavaComponentUtil.parseDependencies(refsOrIncludes);
			for (String cn : deps) {
				IResourceHandle rh = IComponentsLookup.INSTANCE.lookup(cn, IResourceHandle.class);
				IJavaComponent jc = rh.as(IJavaComponent.class);
				if (jc == null) {
					if (!includes) {
						throw new IllegalStateException("Java component dependency resolution error. Failed to resolve component " + cn+ ": " + this.desc.getName());
					}
					if (rh.as(File.class)==null) {
						// not even a files component. Unacceptable
						throw new IllegalStateException("Component dependency resolution error. Failed to resolve component (neither as File nor as Java component) " + cn+ ": " + JavaComponentImpl.this.desc.getName());
					}
				} else {
					javaDeps.add(jc);
				}
				handle().addDependency(rh);
			}
		}
		return javaDeps;
	}
		
	/**
	 * setup class loaders 
	 * @param pubrefs
	 * @param privrefs
	 * @param cmproot
	 * @param desc
	 * @param devMode
	 * @throws IOException
	 * @throws MalformedURLException
	 */
	public void setupClassLoaders(
			List<IJavaComponent> pubrefs, 
			List<IJavaComponent> privrefs,
			// for testing, the runtime folder is passed on
			File cmpRoot,
			IComponentDescriptor desc, 
			boolean devMode
	) throws IOException, MalformedURLException {
		// set up class loaders.
		// public loader
		ClassLoader[] cls = new ClassLoader[pubrefs.size()+ 1];
		int i=0;
		for (IJavaComponent jc : pubrefs) {
			cls[i++] = jc.getPublicLoader();
		}
		cls[i] = PARENT_LOADER;
		// conditional classpath inclusion
		Pattern clpIncl = Pattern.compile(desc.getProperties().getProperty(PUBLIC_CLASSPATH_PATTERN,DEFAULT_CLASSPATH_PATTERN));
		JavaComponentImpl.this.publloader = new ComponentClassLoader(
			handle(), 
			JavaComponentImpl.this.name + "/public", 
			_urls(clpIncl,new File(cmpRoot, BIN_API),new File(cmpRoot,BIN)), 
			cls
		);

		// private loader
		cls = new ClassLoader[privrefs.size()+ 1];
		i=0;
		for (IJavaComponent jc : privrefs) {
			cls[i++] = jc.getPublicLoader();
		}
		cls[i] = JavaComponentImpl.this.publloader;
		
		// conditional classpath inclusion
		clpIncl = Pattern.compile(desc.getProperties().getProperty(PRIVATE_CLASSPATH_PATTERN,DEFAULT_CLASSPATH_PATTERN));
		URL[] clp = _urls(clpIncl,new File(cmpRoot, BIN_IMPL),new File(cmpRoot, BIN_PRIVATE));
		if (devMode) {
			// add test classpath
			clpIncl = Pattern.compile(desc.getProperties().getProperty(TEST_CLASSPATH_PATTERN,DEFAULT_CLASSPATH_PATTERN));
			URL[] test = _urls(clpIncl,new File(cmpRoot,BIN_TEST));
			// append
			if (test.length>0) {
				clp = Arrays.copyOf(clp,clp.length+test.length);
				System.arraycopy(test, 0, clp, clp.length-test.length, test.length);
			}
		}
		
		JavaComponentImpl.this.privloader = new ComponentClassLoader(
			handle(), 
			JavaComponentImpl.this.name + "/private", 
			clp, 
			cls
		);
	}

	private IComponentDescriptor _getDescriptor() {
		if (this.desc == null) {
			this.desc = IComponentsManager.INSTANCE.getComponent(this.name);
		}
		return this.desc;
	}

	// collect class loader urls
	private URL[] _urls(Pattern clpIncl, File ... files) throws MalformedURLException {
		List<URL> us = new ArrayList<URL>();
		for (File file : files) {
			if (file!=null && file.exists() && file.isDirectory()) {
				File f = new File(file, "classes");
				if (f.exists() && f.isDirectory()) {
					us.add(f.toURI().toURL());
				}
				f = new File(file, "lib");
				if (f.exists() && f.isDirectory()) {
					File[] fs = f.listFiles(g->!g.isDirectory() && clpIncl.matcher(g.getName()).matches());
					for (File g : fs) {
						us.add(g.toURI().toURL());
					}
				}
			}
		}
		return us.toArray(new URL[us.size()]);
	}

	// clean state
	private void _clean() {
		try {
			if (this.privloader!=null)
				this.privloader.invalidate();
			if (this.publloader!=null)
				this.publloader.invalidate();
			this.privloader = null;
			this.publloader = null;
			this.desc = null;
		} finally {
			this.inited = false;
		}
	}


	@Override
	public void invalidate()
			throws ResourceBusyException {
		synchronized (this) {
			this.counter = (++COUNTER);
			_clean();
		}
	}

	// ijavacomponent
	public JavaComponentClassLoader getPrivateLoader() {
		return this.privloader;
	}

	public JavaComponentClassLoader getPublicLoader() {
		return this.publloader;
	}
	
	@Override
	public synchronized File getRuntimeResources() {
		return this.runtimeResources;
	}

	protected final static Logger logger = Logger.getLogger(JavaComponentImpl.class.getName());
}
