/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.impl.components.local;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.components.provider.IComponentsRepository;
import com.zfabrik.components.provider.IComponentsRepositoryContext;
import com.zfabrik.util.expression.X;
import com.zfabrik.util.runtime.Foundation;

public class FSRepository implements IComponentsRepository {
	private File repoRoot;

	private Map<String,ComponentImpl> components;
	private Set<String> modules;
	private IComponentsRepositoryContext context;

	public FSRepository(IComponentsManager mgr) throws IOException {
		// compute the repo root
		this.repoRoot = new File(Foundation.getProperties().getProperty(Foundation.HOME_LAYOUT_LOCAL));
		if (!this.repoRoot.exists())
			throw new IllegalStateException("No local repository at "+this.repoRoot.getAbsolutePath());

		final Map<String,ComponentImpl> components =  new HashMap<String, ComponentImpl>();
		final Set<String> modules = new HashSet<String>();
		
		this.repoRoot.listFiles(new FileFilter() {
			public boolean accept(File pathname) {
				if ((pathname.isDirectory()) && (!pathname.getName().startsWith("."))) {
					ComponentImpl.scan(pathname, components, modules);
				}
				return false;
			}
		});
		this.components = Collections.unmodifiableMap(components);
		this.modules = Collections.unmodifiableSet(modules);
		this.context = mgr.registerRepository(1000, this);
	}


	@Override
	public Collection<String> findComponents(X propertyExpression) throws IOException {
		return findComponents(propertyExpression, false);
	}

	@Override
	public Collection<String> findComponents(X propertyExpression, boolean localOnly) throws IOException {
		Collection<String> cs;  

		IComponentsRepository r = this.context.next();
		if (r!=null && ! localOnly) {
			cs = r.findComponents(propertyExpression);
			cs.removeAll(this.components.keySet());
		} else {
			cs = new HashSet<String>();
		}
		Object o;
		for (ComponentImpl ci:this.components.values()) {
			// Search is on raw properties!
			o = propertyExpression.eval(ci.getRawProperties());
			if ((o instanceof Boolean) && ((Boolean) o).booleanValue()) {
				cs.add(ci.getName());
			}
		}
		return cs;
	}

	public IComponentDescriptor getComponent(String component) {
		return getComponent(component, false);
	}

	@Override
	public IComponentDescriptor getComponent(String component, boolean localOnly) {
		IComponentDescriptor res = this.components.get(component);
		if (res != null) {
			return res;
		}
		
		if (!localOnly) {
			IComponentsRepository r = this.context.next();
			if (r != null) {
				res = r.getComponent(component);
			}
		}
		return res;
	}

	// always 0, unless we don't have it... falls back to -1 
	public long getRevision(String component) throws IOException {
		return getRevision(component, false);
	}

	@Override
	public long getRevision(String component, boolean localOnly) throws IOException {
		if (this.components.containsKey(component)) {
			return 0;
		}
		
		if (! localOnly) {
			IComponentsRepository r = this.context.next();
			if (r!=null) {
				return r.getRevision(component);
			}
		}
		return -1;
	}	

	public File retrieve(String name) throws IOException {
		return retrieve(name, false);
	}

	@Override
	public File retrieve(String name, boolean localOnly) throws IOException {
		if (this.components.containsKey(name)) {
			return new File(this.repoRoot,name);
		}
		
		if (! localOnly) {
			IComponentsRepository r = this.context.next();
			if (r!=null) {
				return r.retrieve(name);
			}
		}
		return null;
	}

	@Override
	public Set<String> getModules() throws IOException {
		return getModules(false);
	}
	
	@Override
	public Set<String> getModules(boolean localOnly) throws IOException {
		if (context.next()==null || localOnly) {
			return this.modules;
		} else {
			HashSet<String> s = new HashSet<String>(context.next().getModules(false));
			s.addAll(this.modules);
			return s;
		}
	}
	//	private final static Logger logger = Logger.getLogger(FSRepository.class.getName());
}
