/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.launch.impl;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.logging.Logger;

/**
 * Utility super class for Z2 command line tools
 */
public abstract class Z2CommandLine {
	protected static final String HELP = "-?";
	private Map<String,String> params;
	private Map<String,String> options;
	protected Logger log;
	private File z2Home;

	public Z2CommandLine() {
		this.log = Logger.getLogger(this.getClass().getName());
		this.z2Home = ProcessRunnerImpl.getZ2Home();
	}
	
	public Z2CommandLine(String[] args) {
		this();
		parseInitialArgs(args);
	}
	
	public Map<String, String> getParams() {
		return params;
	}
	
	public Map<String, String> getOptions() {
		return options;
	}
	
	public File getZ2Home() {
		return z2Home;
	}
	
	private void parseInitialArgs(String[] args) {
		try {
			this.params = parseArgs(Arrays.asList(args));
			if (params.containsKey(HELP)) {
				usage();
			}
			this.options = parseOptions(Arrays.asList(args));
		} catch (IllegalArgumentException iae) {
			this.log.severe(iae.getMessage());
			usage();
		}
	}
	
	public static Map<String,String> parseArgs(List<String> args) {
		Map<String,String> p = new HashMap<>();
		parseArgs(args, p::put);
		return p;
	}
	
	/**
	 * Unlike the args, the options are only those args with a hyphen,
	 * and the value part is the identified value.
	 */
	public static Map<String,String> parseOptions(List<String> args) {
		Map<String,String> p = new HashMap<>();
		parseArgs(args, (n,v)->{
			if (v.startsWith(n)) {
				v=v.substring(n.length());
				if ((v=v.trim()).length()>0) {
					p.put(n, v);
				}
			}
		});
		return p;
	}

	
	/**
	 * Parse arguments. Anything starting with a hyphen is considered
	 * an option. In that any "=" or ":" is considered a name-value separator,
	 * also, anything that does not start with a hyphen is considered a 
	 * value for the previous option (if it did not have a value).
	 *
	 * In the params, we pass the identified "option key" and the complete option value. That is, for example:
	 * 
	 * <p>-Dx=a gives (-Dx,-Dx=a)</p>
	 * <p>y=a gives (y=a,y=a)</p>
	 * <p>-z u gives (-z,-z u)</p>
	 * <p> u gives (u,u)</p>
	 * 
	 */
	public static void parseArgs(List<String> args, BiConsumer<String,String> paramFound) {
		String lastOptionWithoutValue = null;
		for (String a : args) {
			if (a.startsWith("-")) {
				// it's an option
				// commit any pending option
				if (lastOptionWithoutValue!=null) {
					paramFound.accept(lastOptionWithoutValue, lastOptionWithoutValue);
					lastOptionWithoutValue=null;
				}
				
				int e = a.indexOf('=');
				int c = a.indexOf(':');
				int s;
				// pick the lowest that is >=0
				if (e>=0) {
					if (c>=0) {
						s = Math.min(e, c);
					} else {
						s = e;
					}
				} else {
					if (c>=0) {
						s=c;
					} else {
						s=-1;
					}
				}
				if (s>=0) {
					// name-value
					paramFound.accept(a.substring(0,s).trim(), a);
				} else {
					// only an option, defer value decision
					lastOptionWithoutValue=a.trim();
				}
			} else {
				// not an option
				if (lastOptionWithoutValue!=null) {
					// a value to a previous option
					paramFound.accept(lastOptionWithoutValue,lastOptionWithoutValue+" "+a);
					lastOptionWithoutValue=null;
				} else {
					// a lone option with no value
					paramFound.accept(a,a);
				}
			}
		}
		// commit any pending option
		if (lastOptionWithoutValue!=null) {
			paramFound.accept(lastOptionWithoutValue, lastOptionWithoutValue);
			lastOptionWithoutValue=null;
		}
	}
	
	
	/**
	 * Show usage for Main class
	 */
	protected void usage() {
		usage(null);
	}
	
	protected void usage(String error) {
		if (error!=null) {
			System.err.println(error);
			System.err.println();
		}
		try (InputStream uin = this.getClass().getResourceAsStream(this.getClass().getSimpleName()+".txt")){
			if (uin==null) {
				log.warning("No usage info found for "+this.getClass().getSimpleName());
			} else {
				System.err.print(new String(uin.readAllBytes(),StandardCharsets.UTF_8));
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
		System.exit(1);
	}

	/**
	 * Rules to compute a Z2
	 */

	
}
