/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.resources;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;

/**
 * A simple type reference to the be used to convey generic type information to type safe APIs such as 
 * {@link IResourceLookup#lookup(String, com.zfabrik.resources.TypeRef)}. Based on the approach outlined in 
 * <a href="http://gafter.blogspot.com/2006/12/super-type-tokens.html">Neil Gafter's blog</a>.
 */
public abstract class TypeRef<T> {
	
	private Type type;
	private Class<T> clz;
	
	@SuppressWarnings("unchecked")
	public TypeRef() {
		Type superclass = this.getClass().getGenericSuperclass();
		if (superclass instanceof Class) {
			throw new RuntimeException("Missing type parameter.");
		}
		this.type = ((ParameterizedType) superclass).getActualTypeArguments()[0];
		if (this.type instanceof Class) {
			clz=(Class<T>)type;
		}
	}

	// used internally
	protected TypeRef(Class<T> clz) {
		this.type = clz;
		this.clz = clz;
	}

	/**
	 * Return the type of the type parameter.
	 */
	public Type getType() {
		return type;
	}
	
	/**
	 * Returns the represented class type, if this is what is represented. Otherwise return <code>null</code> 
	 */
	public Class<T> getClz() {
		return getClz(false);
	}
	
	/**
	 * A cast method to T. Simply to avoid having using code handle warning for casts to parameterized types 
	 */
	@SuppressWarnings("unchecked")
	public T cast(Object o) {
		return (T) o;
	}

	/**
	 * Returns the represented class type, if this is what is represented.  
	 * Otherwise return <code>null</code>, if required is passed as <code>false</code>. 
	 * Throws an {@link IllegalStateException}, if no class type is represented and required is set to <code>true</code>.
	 * Returns <code>null</code>, if no class type is represented and required is set to <code>false</code>. 
	 *  
	 */
	public Class<T> getClz(boolean required) {
		if (clz==null && required) {
			throw new IllegalStateException("Not a simple class type reference but instead for "+type);
		}
		return clz;
	}
	
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((clz == null) ? 0 : clz.hashCode());
		result = prime * result + ((type == null) ? 0 : type.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof TypeRef)) 
			return false;
		TypeRef<?> other = (TypeRef<?>) obj;
		if (clz == null) {
			if (other.clz != null)
				return false;
		} else if (!clz.equals(other.clz))
			return false;
		if (type == null) {
			if (other.type != null)
				return false;
		} else if (!type.equals(other.type))
			return false;
		return true;
	}
	
	@Override
	public String toString() {
		return type.toString();
	}

}
