/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.resources.provider;

import com.zfabrik.resources.IResourceHandle;
import com.zfabrik.resources.ResourceBusyException;
import com.zfabrik.resources.TypeRef;


/**
 * Resource base class. 
 * <p>
 * Resources model named, memory-managed objects that may have dependencies onto other resources and 
 * may have a non-trivial life cycle. For example all component factories are required to provide
 * extensions of this class. 
 * <p>
 * The resource management system is the underpinning of z2's on-demand behavior. The typical life cycle of a resource implementation
 * instance looks as follows
 * <ol>
 * <li>Provisioning by name. The resource will be known by the given name from there on.</li>
 * <li>
 * Any number of {@link #as(Class)} calls may occur. In order to provide the required result or to implement some side-effect, a 
 * resource implementation may build up state and use other resources:
 * 
 * <ul>
 * <li>
 * For every resource required the resource at hand should declare dependency calling 
 * <pre>handle().addDependency(&lt;handle of required resource&gt;)</pre> 
 * See also {@link IResourceHandle#addDependency(IResourceHandle)}.
 * </li>
 * 
 * <li>
 * If returned objects are not associated to the resource instance natively, e.g. by using non-static inner classes, 
 * the resource should call 
 * <pre>handle().attach(Object)</pre> 
 * if it is desired that the resource instance be kept as long as the returned objects (see {@link IResourceHandle#attach(Object)}).
 * </li>
 * 
 * <li>State changes should be <code>synchronized</code> as invalidations may happen at any time</li>
 *  
 * <li>
 * A resource may change its management model using 
 * <pre>handle().adjust(long, long, short)</pre>
 * See also {@link IResourceHandle#adjust(long, long, short)}.
 * Reasons for this vary. For example Web Applications require explicit "stop" calls, so that a resource representing a Web application may not be silently collected. 
 * </li>
 * </ul>
 * </li>
 * 
 * <li>
 * Unless managed by hard references (see {@link IResourceHandle#adjust(long, long, short)}), the resource may be collected once no other reference holds on to a resource handle for it.
 * </li>
 * 
 * <li>
 * Resource may be invalidated, most prominently due to synchronization of component repositories. 
 * Invalidation observes dependencies. That is, a resource will be invalidated also if (and before) 
 * any of its dependency resources will be invalidated. 
 * <p>
 * During invalidation, a resource should clean up any state built up since provisioning and return to a freshly provisioned status.
 * </li>
 *  </ol>
 *  
 * Resource is a class rather than an 
 * interface in order provide a simple and safe association of the
 * resource implementation with its handle.
 * 
 * @author hb
 *
 */
public abstract class Resource {
	private IResourceHandle handle;
	
	public final void init(IResourceHandle c) {
		this.handle = c;
		init();
	}
	
	/**
	 * Called at initialization time with the resource management. At the point in time this
	 * method gets called, a handle has been associated. 
	 */
	public void init() {}
	
	/**
	 * Return the handle for this resource. Use as outlined in the classes documentation.
	 */
	public final IResourceHandle handle() {
		return this.handle;
	}
	
	/**
	 * This method gets called whenever a dependency resource has been invalidated or this resource needs
	 * to be invalidated.
	 * <p>
	 * This code should be executed in a life cycle code block of this resource instance, i.e. where dependencies
	 * are effectively managed, so that race conditions can be avoided.
	 * <p>
	 * State changing methods of a resource should always be synchronized (e.g. on this). This is
	 * in particular true for cross-resource dependencies. In order to assure consistency under race conditions, 
	 * a dependent resource should first declare its dependency and then retrieve the resource implementation.
	 * In case of failures, resources should clean up by calling <code>handle().invalidate(true);</code> 
	 * 
	 * @throws ResourceBusyException
	 */
	public void invalidate() throws ResourceBusyException {}
	
	/**
	 * Retrieve a typed representation of the resource.
	 * <p>
	 * 
	 * @param clz expected return type
	 * @return the expected return type instance represented by the resource or <code>null</code>
	 * if the type facade is not supported. 
	 * @throws ResourceTypeNotAvailableException if the type is supported but cannot be made 
	 * available due to an error situation
	 */
	public  <T> T as(Class<T> clz) {	return null;}
	
	
	/**
	 * Retrieve a resource representation adhering to the passed-on type reference. For components default 
	 * resolutions exist. The default implementation delegates to {@link #as(Class)} whenever possible.
	 * 
	 * In order to check for a {@link TypeRef} to match something expected, compare with a local type reference. E.g.
	 * <pre>
	 * <code>
	 * private TypeRef<Supplier<String>> ref = new TypeRef<>(){};
	 * 
	 * public <T> T as(TypeRef<T> typeRef) {
	 *   if (ref.equals(typeRef)) {
	 *     return typeRef.cast((Supplier<String>) ()->"Hello World");
	 *   }
	 *   return super.as(typeRef);
	 * }
	 * </code>
	 * </pre>
	 * 
	 * 
	 * @param typeRef the expected return type specified via {@link TypeRef} (that also supports parameterized types)
	 */
	public <T> T as(TypeRef<T> typeRef) {
		return as(typeRef.getClz(true));
	}

}
