/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.sync;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.components.IComponentDescriptor;
import com.zfabrik.components.IComponentsLookup;
import com.zfabrik.components.IComponentsManager;
import com.zfabrik.resources.IResourceHandle;
import com.zfabrik.resources.IResourceManager;
import com.zfabrik.util.expression.X;
import com.zfabrik.util.sync.ISynchronization;
import com.zfabrik.util.sync.ISynchronizer;
import com.zfabrik.work.IThreadPool;
import com.zfabrik.work.WorkManager;

/**
 * Performs a synchronization of the z2 home. This will look up all synchronizers (see {@link ISynchronizer})
 * and implement the synchronization process over them.
 * 
 * @author hb
 *
 */
public class SynchronizationRunner implements Runnable {
	private static final X X_SYNCHRONIZERS = X.var(IComponentDescriptor.COMPONENT_TYPE).eq(X.val(ISynchronizer.TYPE));
	private static IThreadPool SyTP = WorkManager.get().createThreadPool("Sync&Verify");
	
	static {
		SyTP.setMaxConcurrency(1);
	}

	private static int execCount = 0;
	private static Long lastSyncStarted;
	private static Long lastSyncCompleted;

	public static synchronized boolean sequenceExecuting() {
		return execCount > 0;
	}
	
	public static synchronized Long getLastSyncStarted() {
		return lastSyncStarted;
	}
	
	public static synchronized Long getLastSyncCompleted() {
		return lastSyncCompleted;
	}

	public static synchronized void start() {
		execCount++;
		lastSyncStarted=System.currentTimeMillis();
	}
	public static synchronized void end() {
		execCount--;
		lastSyncCompleted=System.currentTimeMillis();
	}
	
	// ------------- the actual runner --------------------//

	public final static short INVALIDATE_AND_VERIFY = 0;
	public final static short VERIFY_ONLY = 1;
	private SynchronizationLogHandler lh = null;
	private short mode = INVALIDATE_AND_VERIFY;
	private Runnable whenDone;

	private static class NameSyncPair implements Comparable<NameSyncPair> {	
		private int prio;
		private IResourceHandle rh;
		private String cn;

		public NameSyncPair(int prio, String cn, IResourceHandle rh) {
			this.rh = rh;
			this.cn = cn;
			this.prio = prio;
		}
		
		// inverse sorting
		public int compareTo(NameSyncPair o) {
			return this.prio-o.prio;
		}
		
		public String toString() {
			return cn+"(prio: "+prio+")";
		}
		
		public ISynchronizer getSynchronizer() {
			// instantiate the actual instance late as even this may lead to side effects like loading
			// old code revisions that the actual synchronizers should not see.
			return rh.as(ISynchronizer.class);
		}
	}

	public SynchronizationRunner() {
	}

	public SynchronizationRunner(short mode) {
		this.mode = mode;
	}

	public synchronized void setKeepLog(boolean keepLog) {
		if ((keepLog) && (this.lh == null)) {
			this.lh = new SynchronizationLogHandler();
		}
		if (!keepLog) {
			this.lh = null;
		}
	}

	public synchronized List<String> getLog() {
		if (this.lh != null) {
			return this.lh.getLog();
		}
		return null;
	}

	public synchronized void setWhenDone(Runnable whenDone) {
		this.whenDone = whenDone;
	}

	public synchronized void setMode(short mode) {
		this.mode = mode;
	}

	public synchronized short getMode() {
		return mode;
	}

	private List<SynchronizationRunner.NameSyncPair> getSynchronizers() throws IOException {
		Collection<String> s = IComponentsManager.INSTANCE.findComponents(X_SYNCHRONIZERS);
		List<SynchronizationRunner.NameSyncPair> syncs = new LinkedList<SynchronizationRunner.NameSyncPair>();
		if (s.size() > 0) {
			for (String cn : s) {
				IResourceHandle rh = IComponentsLookup.INSTANCE.lookup(cn, IResourceHandle.class);
				IComponentDescriptor cd = rh.as(IComponentDescriptor.class);
				String spr = cd.getProperties().getProperty(ISynchronizer.PRIO);
				if (spr == null) {
					logger.warning("Synchronizer lacks priority (" + ISynchronizer.PRIO + "): " + cn);
				} else {
					try {
						Integer prio = Integer.valueOf(spr.trim());
						syncs.add(new NameSyncPair(prio, cn, rh));
					} catch (NumberFormatException nfe) {
						logger.warning("Synchronizer has invalid priority value (" + spr + "): " + cn);
					}
				}
			}
		}
		Collections.sort(syncs);
		logger.fine("Found "+syncs.size()+" synchronizers: "+syncs);
		return syncs;
	}

	public void run() {
		start();
		SynchronizationLogHandler lh = null;
		try {
			boolean invalidate;
			synchronized (this) {
				invalidate = this.mode == INVALIDATE_AND_VERIFY;
				lh = this.lh;
			}
			if (lh != null) {
				lh.clear();
				lh.setThread(Thread.currentThread());
				Logger.getLogger("").addHandler(lh);
			}
			ISynchronization sy = new SynchronizationImpl();

			// ---------- invalidation step --------------//
			if (invalidate) {
				List<SynchronizationRunner.NameSyncPair> syncs = getSynchronizers();
				// ascending
				for (SynchronizationRunner.NameSyncPair nsp : syncs) {
					logger.fine("Pre-Invalidation step of synchronizer: " + nsp.cn);
					try {
						ISynchronizer s = nsp.getSynchronizer();
						if (s!=null) {
							s.preInvalidation(sy);
						} else {
							logger.log(Level.WARNING, "No synchronizer instance provided by " + nsp.cn);
						}
					} catch (Exception e) {
						logger.log(Level.WARNING, "Error while in pre-invalidation step of synchronizer: " + nsp.cn, e);
					}
				}
				logger.info("Found " + sy.getInvalidationSet().size() + " invalidation candidate resources");
				if (logger.isLoggable(Level.FINE)) {
					if (!sy.getInvalidationSet().isEmpty()) {
						logger.fine("Trying to invalidate "+sy.getInvalidationSet());
					}
				}
				int i = IResourceManager.INSTANCE.invalidate(sy.getInvalidationSet());
				logger.info("Invalidated " + i + " resources");
			}
			// ---------- verification step --------------//
			List<SynchronizationRunner.NameSyncPair> syncs = getSynchronizers();
			// descending
			Collections.reverse(syncs);
			for (SynchronizationRunner.NameSyncPair nsp : syncs) {
				logger.fine("Completion step of synchronizer: " + nsp.cn);
				try {
					ISynchronizer s = nsp.getSynchronizer();
					if (s!=null) {
						s.complete(sy);
					} else {
						logger.log(Level.WARNING, "No synchronizer instance provided by " + nsp.cn);
					}
				} catch (Exception e) {
					logger.log(Level.WARNING, "Error while in completion step of synchronizer: " + nsp.cn, e);
				}
			}
			if (invalidate)
				logger.fine("Completed synchronization");
			else
				logger.fine("Completed verification");
		} catch (Exception e) {
			logger.log(Level.WARNING, "Error while running synchronization", e);
		} finally {
			Runnable wd = null;
			synchronized (this) {
				wd = this.whenDone;
			}
			try {
				if (wd != null) {
					wd.run();
				}
			} finally {
				if (lh != null) {
					Logger.getLogger("").removeHandler(lh);
				}
				end();
			}
		}
	}

	public void execute(boolean wait) {
		SyTP.execute(wait,this);
	}
	
	private final static Logger logger = Logger.getLogger(SynchronizationRunner.class.getName());
}