/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.util.internal;

import java.io.IOException;
import java.util.HashSet;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This is a utility place where worker processes get registered during their start. A stop request at that time will
 * simply kill all starting workers the hard way. Usually this is due to user interactions when workers hang or get
 * started at the time of a desired shutdown
 *
 * @author hb
 *
 */
public class WorkerVault {
	public final static WorkerVault INSTANCE = new WorkerVault();

	private Set<Process> processes = new HashSet<Process>();
	private boolean accepting = true;

	public Process exec(String cmdline) throws IOException {
		synchronized (this) {
			if (!accepting) {
				throw new IllegalStateException("In shutdown: No new processes allowed!");
			}
		}
		Process p = null;
		try {
			p = Runtime.getRuntime().exec(cmdline);
			return p;
		} finally {
			if (p!=null) {
				synchronized (this) {
					this.processes.add(p);
				}
			}
		}
	}

	public Process exec(ProcessBuilder processBuilder) throws IOException {
		synchronized (this) {
			if (!accepting) {
				throw new IllegalStateException("In shutdown: No new processes allowed!");
			}
		}
		Process p = null;
		try {
			p = processBuilder.start();
			return p;
		} finally {
			if (p!=null) {
				synchronized (this) {
					this.processes.add(p);
				}
			}
		}
	}


	public void release(Process p) {
		synchronized (this) {
			this.processes.remove(p);
		}
	}

	public void shutDown() {
		Set<Process> tokill;
		synchronized (this) {
			accepting=false;
			tokill = new HashSet<Process>(this.processes);
		}
		if (tokill.size()>0) {
			logger.info("Emergeny shutdown of ("+tokill.size()+") pending worker processes...");
			for (Process p : tokill) {
				try {
					p.destroy();
				} catch (Exception e) {
					logger.log(Level.WARNING,"Error during emergency shutdown of worker process",e);
				}
			}
		}
	}

	private final static Logger logger = Logger.getLogger(WorkerVault.class.getName());
}