/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.util.logging;

import java.io.IOException;
import java.util.logging.FileHandler;
import java.util.logging.LogManager;

import com.zfabrik.util.runtime.Foundation;


/**
 * This modification of the JUL file handler
 * prepends a node name (home, or worker process component name) to the
 * file handler's file pattern.
 * <p>
 * Otherwise it is a drop-in replacement for {@link FileHandler}. That is, all configurations
 * apply but instead of using the prefix <code>java.util.logging.FileHandler</code> you use the prefix <code>com.zfabrik.util.logging.NodeFileHandler</code>.
 * For example, in order to set a formatter, use <code>com.zfabrik.util.logging.NodeFileHandler.formatter</code>.
 *
 * @author hb
 *
 */
public class NodeFileHandler extends FileHandler {

	/**
	 * This constructor assumes all config from the {@link LogManager}
	 */
	public NodeFileHandler() throws IOException, SecurityException {
		super(getPattern(null), getLimit(), getCount(), getAppend());
	}

	/**
	 * This constructor takes append from the {@link LogManager} and defaults to 0 and 1 for limit and count resp.
	 */
	public NodeFileHandler(String pattern) throws IOException, SecurityException {
		super(getPattern(pattern), 0, 1, getAppend());
	}

	/**
	 * This constructor defaults to 0 and 1 for limit and count resp.
	 */
	public NodeFileHandler(String pattern, boolean append) throws IOException, SecurityException {
		super(getPattern(pattern), 0, 1, append);
	}

	/**
	 * This constructor takes append from the {@link LogManager}.
	 */
	public NodeFileHandler(String pattern, int limit, int count) throws IOException, SecurityException {
		super(getPattern(pattern), limit, count, getAppend());
	}

	/**
	 * Full-configuration constructor
	 */
	public NodeFileHandler(String pattern, int limit, int count, boolean append) throws IOException, SecurityException {
		super(getPattern(pattern), limit, count, append);
	}

	private static String getPattern(String pattern) {
		if (pattern==null) {
			LogManager manager = LogManager.getLogManager();
			pattern = manager.getProperty(NodeFileHandler.class.getName()+ ".pattern");
		}
		if (pattern==null) {
			pattern=".log";
		}
		// augment the pattern
		String nn;
		if (Foundation.isWorker()) {
			nn = System.getProperty(Foundation.PROCESS_WORKER).replace('/', '_');
		} else {
			nn = "home";
		}
		int p = pattern.lastIndexOf('/');
		// prepend node name
		if (p>=0) {
			p++;
			pattern = pattern.substring(0,p)+nn+pattern.substring(p);
		} else {
			pattern = nn+pattern;
		}
		return pattern;
	}

	private static boolean getAppend() {
		String sval = LogManager.getLogManager().getProperty(NodeFileHandler.class.getName()+ ".append");
		if (sval!=null) {
			return Boolean.parseBoolean(sval);
		}
		return false;
	}

	private static int getLimit() {
		return getInt(".limit",0);
	}

	private static int getCount() {
		return getInt(".count",1);
	}

	private static int getInt(String suffix, int def) {
		String sval = LogManager.getLogManager().getProperty(NodeFileHandler.class.getName()+ suffix);
		if (sval!=null) {
			try {
				return Integer.parseInt(sval.trim());
			} catch (NumberFormatException nfe) {
				return def;
			}
		}
		return def;
	}


}
