/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.util.threading;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.sql.SQLException;
import java.util.concurrent.Callable;
import java.util.function.Supplier;

import com.zfabrik.util.function.ThrowingRunnable;
import com.zfabrik.util.function.ThrowingSupplier;

/**
 * Threading utils. Not thread pools and such but rather utilities that help
 * isolating resources and avoiding memory leaks.
 * 
 * @author hb
 *
 */
public class ThreadUtil {

	/**
	 * Stupid... but there is places all over that hold on to
	 * AccessControlContexts and Context Class Loaders. This easily consitutes a
	 * leak. So, for stuff that doesn't need anything from the context class
	 * loader this is a way of keeping context to a minimum;
	 * <p>
	 * Execute a callable with a clean thread context and security context as
	 * to avoid any passing on of thread context and security context to another
	 * thread that may be spawned from here and may end up holding copies in the end.
	 * Any exception thrown will be propagated.
	 * <p>
	 * See also {@link #confine(Callable, Class)}.
	 * 
	 * @param contextLoader A class loader to set as context class loader 
	 * @param callable A {@link Callable} to invoke
	 * @return Return value from the {@link Callable} invocation
	 * @throws Exception 
	 */
	public static <T> T cleanContextExceptionExecute(ClassLoader contextLoader, Callable<T> callable) throws Exception {
		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		Thread.currentThread().setContextClassLoader(contextLoader);
		try {
			try {
				return AccessController.doPrivileged((PrivilegedExceptionAction<T>) ()->callable.call());
			} catch (PrivilegedActionException e) {
				if (e.getCause() instanceof Exception) {
					throw (Exception) e.getCause();
				}
				throw e;
			}
		} finally {
			Thread.currentThread().setContextClassLoader(cl);
		}
	}

	/**
	 * Short version of {@link #cleanContextExceptionExecute(ClassLoader, Callable)} passing in <code>null</code>
	 * as context class loader.
	 * 
	 * @param callable A {@link Callable} to invoke
	 * @return Return value from the {@link Callable} invocation
	 * @throws Exception
	 */
	public static <T> T cleanContextExceptionExecute(final Callable<T> callable) throws Exception {
		return cleanContextExceptionExecute(null, callable);
	}

	/**
	 * Execute a callable with a clean thread context and security context as
	 * to avoid any passing on of thread context and security context to another
	 * thread that may be spawned from here and may end up holding copies in the end.
	 * Any exception caught will be wrapped in a generic runtime exception.
	 * 
	 * @param contextLoader A class loader to set as context class loader 
	 * @param callable A {@link Callable} to invoke
	 * @return Return value from the {@link Callable} invocation
	 */
	public static <T> T cleanContextExecute(ClassLoader contextLoader, final Callable<T> callable) {
		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		Thread.currentThread().setContextClassLoader(contextLoader);
		try {
			return AccessController.doPrivileged((PrivilegedAction<T>) ()-> {
				try {
					return callable.call();
				} catch (Exception e) {
					throw new RuntimeException(e);
				}
			});
		} finally {
			Thread.currentThread().setContextClassLoader(cl);
		}
	}
	
	/**
	 * Short version of {@link #cleanContextExecute(ClassLoader, Callable)} passing in <code>null</code>
	 * as context class loader.
	 * 
	 * @param callable A {@link Callable} to invoke
	 * @return Return value from the {@link Callable} invocation
	 */
	public static <T> T cleanContextExecute(final Callable<T> callable) {
		return cleanContextExecute(null, callable);
	}
	
	/**
	 * Execute a {@link Supplier} with a clean thread context and security context as
	 * to avoid any passing on of thread context and security context to another
	 * thread that may be spawned from here and may end up holding copies in the end.
	 * 
	 * @param contextLoader A class loader to set as context class loader 
	 * @param supplier A {@link Supplier} to invoke
	 * @return Return value from the {@link Supplier} invocation
	 */
	public static <T> T cleanContextSupply(ClassLoader contextLoader, Supplier<T> supplier) {
		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		Thread.currentThread().setContextClassLoader(contextLoader);
		try {
			return AccessController.doPrivileged((PrivilegedAction<T>) ()-> supplier.get());
		} finally {
			Thread.currentThread().setContextClassLoader(cl);
		}
	}
	
	/**
	 * Short version of {@link #cleanContextSupply(ClassLoader, Supplier)} passing in <code>null</code>
	 * as context class loader.
	 * 
	 * @param supplier A {@link Supplier} to invoke
	 * @return Return value from the {@link Supplier} invocation
	 */
	public static <T> T cleanContextSupply(Supplier<T> supplier) {
		return cleanContextSupply(null, supplier);
	}

	/**
	 * Using this method, an invocation of a callable may be wrapped to only throw a declared
	 * checked exception.
	 * <p>
	 * <ul>
	 * <li>If the supplied invocation throws the confinement exception or a sub class, this exception will be rethrown.</li>
	 * <li>If the supplied invocation throws the {@link RuntimeException}, this exception will be rethrown. </li>
	 * <li>Any other exception will be wrapped into a {@link RuntimeException}</li>
	 * <p>
	 * This is handy when it is required to pass a callable given an execution that potentially throws a check exception that 
	 * is much narrower in scope than simply {@link Exception}.
	 * <p>
	 * For example given a method <code>readTable</code> that throws a {@link SQLException}, an equivalent execution using
	 * {@link ThreadUtil#cleanContextExceptionExecute(Callable)} looks like this:
	 * <pre><code>
	 * confine(()->cleanContextExceptionExecute(this::readTable),SQLException.class);
	 * </code></pre>
	 * 
	 * @param eclz the confinement exception
	 * @param call the {@link Callable} to invoke
	 * @return the result of the {@link Callable} invocation
	 * @throws E A confinement exception
	 */
	public static <T,E extends Exception> T confine(Callable<T> call,Class<E> eclz) throws E {
		try {
			return call.call();
		} catch (Exception e) {
			if (eclz.isAssignableFrom(e.getClass())) {
				throw eclz.cast(e);
			}
			if (e instanceof RuntimeException) {
				throw (RuntimeException)e;
			}
			throw new RuntimeException(e);
		}
	}

	/**
	 * Variation of {@link #cleanContextExceptionExecute(ClassLoader, Callable)} that relies on {@link ThrowingSupplier} to
	 * generically limit the amount of noise due to missing exception interference. In many places using lambdas this approach
	 * is syntactically nicer than using confine 
	 * 
	 */
	@SuppressWarnings("unchecked")
	public static <T,E extends Exception> T cleanContextGet(ClassLoader contextLoader, ThrowingSupplier<T, E> supplier) throws E {
		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		Thread.currentThread().setContextClassLoader(contextLoader);
		try {
			try {
				return AccessController.doPrivileged((PrivilegedExceptionAction<T>) ()->supplier.get());
			} catch (PrivilegedActionException e) {
				Throwable cause = e.getCause();
				if (cause instanceof RuntimeException) {
					throw (RuntimeException) cause;
				}
				if (cause instanceof Exception) {
					// must be our E 
					throw (E) cause;
				}
				throw new RuntimeException(e);
			}
		} finally {
			Thread.currentThread().setContextClassLoader(cl);
		}
	}
	
	/**
	 * Variation of {@link #cleanContextExceptionExecute(Callable)} that relies on {@link ThrowingSupplier} to
	 * generically limit the amount of noise due to missing exception interference. In many places using lambdas this approach
	 * is syntactically nicer than using confine 
	 * 
	 */
	public static <T,E extends Exception> T cleanContextGet(ThrowingSupplier<T, E> supplier) throws E {
		return cleanContextGet(null,supplier);
	}
	
	/**
	 * Execute a {@link ThrowingRunnable} with a clean thread context and security context as
	 * to avoid any passing on of thread context and security context to another
	 * thread that may be spawned from here and may end up holding copies in the end.
	 * 
	 * @param contextLoader A class loader to set as context class loader 
	 * @param run A {@link Runnable} to invoke
	 */
	public static <E extends Exception> void cleanContextRun(ClassLoader contextLoader, final ThrowingRunnable<E> run) throws E {
		cleanContextGet(
			contextLoader, 
			()->{
				run.run();
				return null;
			}
		);
	}
	
	/**
	 * Short version of {@link #cleanContextRun(ClassLoader, Runnable)} passing in <code>null</code>
	 * as context class loader.
	 * 
	 * @param run A {@link Runnable} to invoke
	 */
	public static <E extends Exception> void cleanContextRun(final ThrowingRunnable<E> run) throws E {
		cleanContextRun(null, run);
	}


}
