/*
 * z2-Environment
 * 
 * Copyright(c) ZFabrik Software GmbH & Co. KG
 * 
 * contact@zfabrik.de
 * 
 * http://www.z2-environment.eu
 */
package com.zfabrik.work;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.zfabrik.util.function.ThrowingSupplier;

/**
 * A Work Unit comprises resources that are bound to some logical work process.
 * 
 * A work unit may be transactional or immediate. Resources are encouraged to
 * register with the work unit.
 * 
 * In order to orderly use the work unit mechanism, it is required to initialize
 * and later close a work unit by using it as an {@link AutoCloseable} or explicitly:
 * <p>
 * <pre>
 * try (WorkUnit.initCurrent()) {
 * ...
 * }
 * </pre>
 * </p>
 * 
 * or 
 * 
 * <p>
 * <pre>
 * WorkUnit.initCurrent();
 * try {
 * ...
 * } finally {
 *    WorkUnit.closeCurrent();
 * }
 * </pre>
 * </p>
 * This pattern is implemented by the {@link #work(Callable)} method and its sibling.
 * 
 * The application thread pool implementation uses this pattern, so that in
 * general it is not necessary to implement it individually, except work has to
 * be done in a Timer Thread and will not be associated with the Application
 * Thread Pool (which is the preferred method!). See also
 * {@link IThreadPool#executeAs(Runnable, boolean)}.
 * 
 * On the current thread however, after any completion, i.a. after a call to close() we release
 * the work unit association. This is for resource consumption reasons (and not to hold on to something
 * that should be collectable) but also so that we can normally see whether a thread
 * has been setup correctly during getCurrent(). We want to know (by exception) if a thread
 * is querying for a work unit although it has not been properly initialized - which in 
 * turn indicates that it will most likely not be completed properly either. 
 * 
 * 
 * @author Henning
 * 
 */
public final class WorkUnit implements AutoCloseable {
	private static ThreadLocal<WorkUnit> current = new ThreadLocal<WorkUnit>();
	private boolean rollbackOnly = false;
	private short nesting = 0;

	/**
	 * queries current work unit but will not instantiate one.
	 */
	public static final WorkUnit queryCurrent() {
		return (WorkUnit) current.get();
	}

	/**
	 * gets current work unit. If none bound, throw an exception as this
	 * indicates that this is a thread that has not orderly initialized the work
	 * unit.
	 */
	public static final WorkUnit getCurrent() {
		WorkUnit unit = queryCurrent();
		if (unit == null) {
			throw new IllegalStateException("No Work Unit found: Missing initialization of work on thread!");
		}
		return unit;
	}

	/**
	 * commits the current work unit (if any)
	 * 
	 */
	public static void commitCurrent() {
		WorkUnit current = queryCurrent();
		if (current != null) {
			current.commit();
		}
	}

	/**
	 * rolls back the current work unit (if any)
	 * 
	 */
	public static void rollBackCurrent() {
		WorkUnit current = queryCurrent();
		if (current != null) {
			current.rollback();
		}
	}

	/**
	 * initializes a work unit orderly on a thread join the current, if present.
	 * 
	 */
	public static WorkUnit initCurrent() {
		WorkUnit wu = queryCurrent();
		if (wu == null) {
			wu = new WorkUnit();
			attach(wu);
		} else {
			wu.nesting++;
		}
		return wu;
	}

	/**
	 * Closes the current work unit (if any). See {@link #close()}.
	 * 
	 */
	public static void closeCurrent() {
		WorkUnit cw = queryCurrent();
		if (cw != null) {
			cw.close();
		}
	}

	/**
	 * Sets the current work unit, if any, to be rolled back on close. See {@link #setRollbackOnly()}
	 */
	public static void setRollbackOnlyCurrent() {
		WorkUnit current = queryCurrent();
		if (current != null) {
			current.setRollbackOnly();
		}
	}

	/**
	 * Gets whether the current work unit, if any, is to be rolled back on close. See {@link #getRollbackOnly()}.
	 */
	public static boolean getRollbackOnlyCurrent() {
		WorkUnit current = queryCurrent();
		if (current != null) {
			return current.getRollbackOnly();
		}
		return false;
	}

	/**
	 * Detaches current work unit (if any) from the current thread
	 */
	public static WorkUnit detach() {
		WorkUnit wu = queryCurrent();
		if (wu!=null) {
			logger.fine("Detached Work Unit "+wu+" from the current thread");
		}
		current.set(null);
		return wu;
	}

	/**
	 * attaches a given work unit with the current thread. Will fail if there is
	 * one currently associated with the current thread.
	 */
	public static void attach(WorkUnit unit) {
		logger.fine("Attached Work Unit "+unit+" to the current thread");
		current.set(unit);
	}

	//
	// non-static
	//
	private Map<String, IWorkResource> resources = new HashMap<String, IWorkResource>(1);

	//
	// no external instantiation
	//
	private long initTime;

	private WorkUnit() {
		this.initTime = System.currentTimeMillis();
	}

	/**
	 * Commit work unit. If the work unit is set for rollback only however, the work unit will be rolled back.
	 * In the case of a commit, all enlisted resources will first receive a {@link IWorkResource#beforeCompletion(boolean)}.
	 * If during that phase the work unit is set to rollback only, a rollback is performed. See {@link #rollback()}.
	 * Otherwise all resources receive a {@link IWorkResource#commit()} before all resources receive a {@link IWorkResource#afterCompletion(boolean)}.
	 */
	public void commit() {
		if (this.rollbackOnly) {
			this.rollback();
		} else {
			logger.finer("Committing Work Unit " + this);
			// go and let all players do before commit work...(but only if we indeed commit)
			_beforeCompletion(this.rollbackOnly);
			// after before completion, the rollback state may have changed
			if (this.rollbackOnly) {
				this.rollback();
			} else {
				try {
					long start = System.currentTimeMillis();
					// go and let all players commit...
					visitResources(this::_commit); 
					long duration = System.currentTimeMillis() - start;
					logger.fine("Committed Work Unit "+this);
					perfLogger.finer("system.workunit,commit," + this + ",duration," + duration + ",ms");
				} finally {
					// after completion - in any case
					_afterCompletion(this.rollbackOnly);
				}
			}
		}
	}
	

	/**
	 * Rollback the work unit. For all enlisted resources a rollback and an after completion event will be triggered. See {@link IWorkResource} for more details.
	 */
	public void rollback() {
		logger.finer("Rolling back Work Unit "+this);
		long start = System.currentTimeMillis();
		try {
			// go and let all players rollback...
			visitResources(this::_rollback);
		} finally {
			logger.fine("Rolled back Work Unit "+this);
			long duration = start - System.currentTimeMillis();
			perfLogger.finer("system.workunit,rollback," + this + ",duration," + duration + ",ms");
			boolean rb = this.rollbackOnly; 
			this.rollbackOnly = false;
			// after completion - in any case
			_afterCompletion(rb);
		}
	}

	/**
	 * Sets this unit to be rolled back on close. That is, instead of committing a close will {@link #rollback()}, if set to <code>true</code>
	 */
	public void setRollbackOnly() {
		if (logger.isLoggable(Level.FINER)) {
			try {
				throw new RuntimeException("RollBackOnly Backtrace");
			} catch (Exception e) {
				logger.log(Level.FINER, "intercepted setRollbackOnly", e);
			}
		}
		this.rollbackOnly = true;
	}

	/**
	 * Gets whether this unit is to be rolled back on close
	 */
	public boolean getRollbackOnly() {
		return this.rollbackOnly;
	}

	/**
	 * Close work unit. Note that a call to {@link #initCurrent()} is either opening a new work unit of joining an existing. In the latter case, a use counter is incremented.
	 * A call to close only performs the actual close action, when the use counter is 1. Otherwise the use counter will simply be decreased.
	 * The actual close action is {@link #rollback()}, if the work unit was set to rollback only (via {@link #setRollbackOnly()} or {@link #setRollbackOnlyCurrent()}) or 
	 * {@link #commit()} otherwise. In both cases, all enlisted resources will receive a close event. See {@link IWorkResource} for more details.
	 * If this work unit is the current work unit, it will be detached and there will be no current work unit anymore.
	 */
	public void close() {
		if (this.nesting>0) {
			this.nesting--;
			return;
		}
		logger.finer("Closing work unit");
		try {
			this.commit();
		} catch (Exception e) {
			logger.log(Level.WARNING, "Problem during commit of Work Unit "+this, e);
		} finally {
			_closeResources();
			long duration = System.currentTimeMillis() - this.initTime;
			logger.fine("Closed work unit");
			perfLogger.fine("system.workunit,use," + this+ ",duration," + duration + ",ms");
			if (this==queryCurrent()) {
				detach();
			}
		}
	}

	private void _closeResources() {
		try {
			visitResources(new Visitor() {
				public void visit(String key, IWorkResource resource) {
					_close(key, resource);
				}

			});
		} finally {
			this.resources.clear();
			logger.finer("Cleared resources");
		}
	}

	private void _close(String key, IWorkResource resource) {
		try {
			resource.close();
		} catch (Exception we) {
			logger.log(Level.WARNING, "error during close of \"" + key + "\"", we);
		} finally {
			logger.finer("Closed \"" + key + "\"");
		}
	}
	/**
	 * bind and, if unit is stateful, begin resource
	 */
	public void bindResource(String key, IWorkResource resource) {
		logger.finer("Attaching resource \"" + key + "\"");
		this.resources.put(key, resource);
		_begin(key, resource);
	}

	/**
	 * unbind and if unit stateful rolls back resource
	 */
	public IWorkResource unbindResource(String key) {
		IWorkResource resource = (IWorkResource) this.resources.remove(key);
		if (resource!=null) {
			logger.finer("Detaching resource \"" + key + "\"");
			_close(key, resource);
		}
		return resource;
	}

	/**
	 * returns a named work resource
	 */
	public IWorkResource getResource(String key) {
		return (IWorkResource) this.resources.get(key);
	}
	
	/**
	 * Execute a runnable within a complete work unit, incl. initialization and close. If the execution of the runnable throws an exception,
	 * the work unit will be rolled back.
	 */
	public static void work(Runnable r) {
		supply((ThrowingSupplier<Void,RuntimeException>) ()->{ r.run(); return null; });
	}
	
	/**
	 * Execute a {@link ThrowingSupplier} within a complete work unit, incl. initialization and close. If the execution of the runnable throws an exception,
	 * the work unit will be rolled back.
	 */
	public static <T, E extends Exception> T supply(ThrowingSupplier<T, E> supplier) throws E {
		// recursion joins!
		boolean success = false;
		try {
			WorkUnit.initCurrent();
			T t = supplier.get();
			success = true;
			return t;
		} catch (VirtualMachineError vme) {
			logger.log(Level.SEVERE,"VM Error during work unit",vme);
			System.exit(-1);
			return null;
		} finally {
			if (!success) {
				logger.log(Level.WARNING, "Setting work unit to rollback only due to exception in work unit");
				WorkUnit.setRollbackOnlyCurrent();
			}
			try {
				WorkUnit.closeCurrent();
			} catch (VirtualMachineError vme) {
				logger.log(Level.SEVERE,"VM Error during completion of work unit",vme);
				System.exit(-1);
			} catch (Throwable t) {
				logger.log(Level.WARNING, "Error during completion of work unit", t);
			}
		}
	}

	/**
	 * Execute a {@link Callable} within a complete work unit, incl. initialization and close. If the execution of the runnable throws an exception,
	 * the work unit will be rolled back.
	 */
	public static <T> T work(Callable<T> r) throws Exception {
		return supply((ThrowingSupplier<T,Exception>) ()->r.call());
	}
	
	//
	// helper
	//
	private void _commit(String key, IWorkResource resource) {
		try {
			resource.commit();
		} catch (Exception we) {
			logger.log(Level.WARNING, "error during commit of \"" + key + "\"", we);
			throw new WorkException("error during commit of \"" + key + "\"", we);
		} finally {
			logger.finer("Committed \"" + key + "\"");
		}
	}

	private void _beforeCompletion(final boolean rollback) {
		visitResources((String key, IWorkResource resource) -> _beforeCompletion(key, resource,rollback));
	}

	private void _beforeCompletion(String key, IWorkResource resource, boolean rollback) {
		try {
			resource.beforeCompletion(rollback);
		} catch (Exception we) {
			logger.log(Level.WARNING, "error during beforeCompletion of \"" + key + "\"", we);
			throw new WorkException("error during beforeCompletion of \"" + key + "\"", we);
		} finally {
			logger.finer("BeforeCompletion \"" + key + "\"");
		}
	}

	private void _afterCompletion(final boolean rollback) {
		visitResources((String key, IWorkResource resource)->_afterCompletion(key, resource,rollback));
	}

	private void _afterCompletion(String key, IWorkResource resource, boolean rollback) {
		try {
			resource.afterCompletion(rollback);
		} catch (Exception we) {
			logger.log(Level.WARNING, "error during afterCompletion of \"" + key + "\"", we);
			throw new WorkException("error during afterCompletion of \"" + key + "\"", we);
		} finally {
			logger.finer("AfterCompletion \"" + key + "\"");
		}
	}
	
	private void _rollback(String key, IWorkResource resource) {
		try {
			resource.rollback();
		} catch (Exception we) {
			logger.log(Level.WARNING, "error during cancellation of \"" + key + "\"", we);
		} finally {
			logger.finer("Cancelled \"" + key + "\"");
		}
	}

	private void _begin(String key, IWorkResource resource) {
		try {
			resource.begin(this);
		} catch (Exception we) {
			logger.log(Level.WARNING, "error during begin() on \"" + key + "\"", we);
			throw new WorkException("error during begin() of \"" + key + "\"", we);
		} finally {
			logger.finer("Began \"" + key + "\"");
		}
	}
	
	private static final Logger logger = Logger.getLogger(WorkUnit.class.getName());
	private static final Logger perfLogger = Logger.getLogger("performance." + WorkUnit.class.getName());
	
	//
	// helper
	//
	private interface Visitor {
		void visit(String key, IWorkResource resource);
	}
	
	// visit all resources at least once even if the visit changes the resources key set
	private void visitResources(Visitor visitor) {
		Set<String> done = new HashSet<String>();
		Set<String> rema = new HashSet<String>();
		while (true) {
			rema.clear();
			rema.addAll(this.resources.keySet());
			rema.removeAll(done);
			if (rema.isEmpty()) {
				break;
			}
			for (String k : rema) {
				visitor.visit(k, this.resources.get(k));
				done.add(k);
			}
		}
	}
}
