package tests.binding;

import java.math.BigDecimal;
import java.math.BigInteger;

import junit.framework.Assert;

import org.json.JSONArray;
import org.json.JSONObject;
import org.junit.Test;

import tests.binding.samples.AlternativeSetter;
import tests.binding.samples.ComplexBean;
import tests.binding.samples.ListSetter;
import tests.binding.samples.NumbersBean;
import tests.binding.samples.SimpleBean;


public class JavaBean {

	
	@Test public void simpleJavaBean() {
		JSONObject jso = new JSONObject()
			.put("a", "Hello")
			.put("b",5)
			.put("readOnly", "readOnly");
		
		SimpleBean sb = jso.toJavaBean(SimpleBean.class);
		
		Assert.assertTrue("Result must not be null", null!=sb);
		Assert.assertTrue("Wrong value a="+sb.getA(), "Hello".equals(sb.getA()));
		Assert.assertTrue("Wrong value b="+sb.getB(), 5==sb.getB());
		Assert.assertTrue("ReadOnly should be null: "+sb.getReadOnly(), null==sb.getReadOnly());
	}
	
	@Test public void simpleArrays() {
		{
			JSONArray jsa = new JSONArray().put("0").put(1).put(2.0f);
			Object[] res = jsa.toJavaArray(Object.class);
			
			Assert.assertTrue("Result must not  be null", res!=null);
			Assert.assertTrue("Wrong element 0: "+res[0], "0".equals(res[0]));
			Assert.assertTrue("Wrong element 1: "+res[1], 1==(Integer)res[1]);
			Assert.assertTrue("Wrong element 2: "+res[2], 2.0f == Number.class.cast(res[2]).floatValue());
		}
		{
			JSONArray jsa = new JSONArray()
				.put(new SimpleBean("Erstes",0,"No"))
				.put(new SimpleBean("Zweites",1,"Never"))
				.put(new SimpleBean("Drittes",2,"Ever"));
	
			SimpleBean[] res = jsa.toJavaArray(SimpleBean.class);
			Assert.assertTrue("Result must not  be null", res!=null);
			Assert.assertTrue("Wrong element 0: "+res[0], res[0].getB()==0);
			Assert.assertTrue("Wrong element 1: "+res[1], res[1].getB()==1);
			Assert.assertTrue("Wrong element 2: "+res[2], res[2].getB()==2);
		}
	}
	
	@Test public void deepJavaBeanWithArrays() {
		
		JSONObject o = new JSONObject(
			"{ beans:["+
				"{a:'Erstes',b:0,readOnly:'No'},"+
				"{a:'Zweites',b:1,readOnly:'Never'},"+
				"{a:'Drittes',b:2,readOnly:'Ever'}"+
			"], "+
			" nested: { number: 3 },"+
			" number: 5"+
			"}");
		
		ComplexBean cb = o.toJavaBean(ComplexBean.class);
		Assert.assertTrue("Result must not be null", null!=cb);
		
		SimpleBean[] res = cb.getBeans(); 
		Assert.assertTrue("Result must not  be null", res!=null);
		Assert.assertTrue("Wrong element 0: "+res[0], res[0].getB()==0);
		Assert.assertTrue("Wrong element 1: "+res[1], res[1].getB()==1);
		Assert.assertTrue("Wrong element 2: "+res[2], res[2].getB()==2);
		
		Assert.assertTrue("Wrong number: "+cb.getNumber(), cb.getNumber()==5);
		Assert.assertTrue("Nested complex bean must not be null", cb.getNested()!=null);
		Assert.assertTrue("Wrong nested number: "+cb.getNested().getNumber(), cb.getNested().getNumber()==3);
		
	}
	
	@Test public void numberConversions() {
		JSONObject o1 = new JSONObject(
				"{ bigDecimal: \"1.23\", bigInteger: \"123\", longNumber: 123, intNumber:123, floatNumber:1.25, doubleNumber:1.25}" 
		);
		
		NumbersBean nb = o1.toJavaBean(NumbersBean.class);
		
		Assert.assertTrue("Result must not  be null", nb!=null);
		Assert.assertTrue("wrong big decimal: "+nb.getBigDecimal(), new BigDecimal("1.23").equals(nb.getBigDecimal()));
		Assert.assertTrue("wrong big integer: "+nb.getBigInteger(), new BigInteger("123").equals(nb.getBigInteger()));
		Assert.assertTrue("wrong long: "+nb.getLongNumber(), nb.getLongNumber()==123);
		Assert.assertTrue("wrong int: "+nb.getIntNumber(), nb.getIntNumber()==123);
		Assert.assertTrue("wrong float: "+nb.getFloatNumber(), nb.getFloatNumber()==1.25f);
		Assert.assertTrue("wrong double: "+nb.getDoubleNumber(), nb.getDoubleNumber()==1.25d);
		
		// BigDecimal, BigInteger to String
		JSONObject jso = new JSONObject(nb);
		Assert.assertTrue("BigDecimal not serialized as string in "+jso.toString(), jso.toString().indexOf("\"bigDecimal\":\"1.23\"")>=0);
		Assert.assertTrue("BigInteger not serialized as string in "+jso.toString(), jso.toString().indexOf("\"bigInteger\":\"123\"")>=0);
		
		// round tripping
		String s = new JSONObject(nb).toString();
		nb = new JSONObject(s).toJavaBean(NumbersBean.class);
		
		Assert.assertTrue("Result must not  be null", nb!=null);
		Assert.assertTrue("wrong big decimal: "+nb.getBigDecimal(), new BigDecimal("1.23").equals(nb.getBigDecimal()));
		Assert.assertTrue("wrong big integer: "+nb.getBigInteger(), new BigInteger("123").equals(nb.getBigInteger()));
		Assert.assertTrue("wrong long: "+nb.getLongNumber(), nb.getLongNumber()==123);
		Assert.assertTrue("wrong int: "+nb.getIntNumber(), nb.getIntNumber()==123);
		Assert.assertTrue("wrong float: "+nb.getFloatNumber(), nb.getFloatNumber()==1.25f);
		Assert.assertTrue("wrong double: "+nb.getDoubleNumber(), nb.getDoubleNumber()==1.25d);
		
	}
	
	@Test public void alternativeSetters() {
		JSONObject o = new JSONObject("{ list: [\"eins\",\"zwei\",\"drei\"]}");
		AlternativeSetter as = o.toJavaBean(AlternativeSetter.class);
		Assert.assertTrue("Result must not  be null", as!=null);
		Assert.assertTrue("List is null",as.getList()!=null);
		Assert.assertTrue("wrong number of entries in list: "+as.getList().size(),as.getList().size()==3);
	}
	
	@Test public void listSetters() {
		JSONObject o = new JSONObject("{ list: [\"eins\",\"zwei\",\"drei\"]}");
		ListSetter as = o.toJavaBean(ListSetter.class);
		Assert.assertTrue("Result must not  be null", as!=null);
		Assert.assertTrue("List is null",as.getList()!=null);
		Assert.assertTrue("wrong number of entries in list: "+as.getList().size(),as.getList().size()==3);
	}
}
